/*
 * SonarQube Python Plugin
 * Copyright (C) 2011-2019 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.python.tree;

import com.sonar.sslr.api.AstNode;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.sonar.python.api.tree.Expression;
import org.sonar.python.api.tree.Name;
import org.sonar.python.api.tree.Parameter;
import org.sonar.python.api.tree.Token;
import org.sonar.python.api.tree.TreeVisitor;
import org.sonar.python.api.tree.TypeAnnotation;
import org.sonar.python.api.tree.Tree;

public class ParameterImpl extends PyTree implements Parameter {

  private final Name name;
  private final TypeAnnotation annotation;
  private final Token equalToken;
  private final Expression defaultValue;
  private final Token starToken;

  public ParameterImpl(AstNode node, @Nullable Token starToken, Name name, @Nullable TypeAnnotation annotation,
                       @Nullable Token equalToken, @Nullable Expression defaultValue) {
    super(node);
    this.starToken = starToken;
    this.name = name;
    this.annotation = annotation;
    this.equalToken = equalToken;
    this.defaultValue = defaultValue;
  }

  @CheckForNull
  @Override
  public Token starToken() {
    return starToken;
  }

  @Override
  public Name name() {
    return name;
  }

  @CheckForNull
  @Override
  public TypeAnnotation typeAnnotation() {
    return annotation;
  }

  @CheckForNull
  @Override
  public Token equalToken() {
    return equalToken;
  }

  @CheckForNull
  @Override
  public Expression defaultValue() {
    return defaultValue;
  }

  @Override
  public Kind getKind() {
    return Kind.PARAMETER;
  }

  @Override
  public void accept(TreeVisitor visitor) {
    visitor.visitParameter(this);
  }

  @Override
  public List<Tree> children() {
    return Stream.of(name, annotation, equalToken, defaultValue, starToken).collect(Collectors.toList());
  }
}
