package org.sqlproc.engine;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sqlproc.engine.impl.SqlEmptyMonitor;
import org.sqlproc.engine.impl.SqlMappingRule;
import org.sqlproc.engine.impl.SqlMetaStatement;

/**
 * Common ancestor for {@link SqlQueryEngine} and {@link SqlCrudEngine}.
 * 
 * <p>
 * For more info please see the Reference Guide or <a
 * href="http://code.google.com/p/sql-processor/w/list">tutorials</a>.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlEngine {

    /**
     * The internal slf4j logger.
     */
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * Name of this META SQL, which uniquely identifies the instance.
     */
    protected String name;

    /**
     * Precompiled META SQL, which is ANSI SQL extension using ANTLR defined grammar.
     */
    protected SqlMetaStatement statement;

    /**
     * Precompiled Mapping rule, which is SQL result to Java output classes mapping prescription.
     */
    protected SqlMappingRule mapping;

    /**
     * Configuration of SQL Processor using Map of features. Optional features alter the SQL Processor behavior.
     */
    protected Map<String, Object> features = new HashMap<String, Object>();

    /**
     * Monitor for the runtime statistics gathering.
     */
    protected SqlMonitor monitor;

    /**
     * Creates a new instance of SqlEngine from one META SQL statement and one SQL Mapping rule instance. Both
     * parameters are already precompiled instances using the ANTLR parsers. This is the recommended usage for the
     * runtime performance optimization. This constructor is devoted to be used from the SqlEngineLoader, which is able
     * to read all definitions from an external queries.properties file and create the named SqlEngine instances. Also
     * an external SQL Monitor for runtime statistics gathering can be engaged.
     * 
     * @param name
     *            the name if this SQL Engine instance
     * @param statement
     *            the precompiled META SQL statement, extension of ANSI SQL
     * @param mapping
     *            the precompiled SQL Mapping rule, SQL result to Java output classes mapping
     * @param monitor
     *            the SQL Monitor for the runtime statistics gathering
     * @param features
     *            the optional SQL Processor features
     */
    public SqlEngine(String name, SqlMetaStatement statement, SqlMappingRule mapping, SqlMonitor monitor,
            Map<String, Object> features) {
        this.name = name;
        this.statement = statement;
        this.mapping = mapping;
        this.features = features;
        this.monitor = (monitor != null) ? monitor : new SqlEmptyMonitor();
    }

}
