package org.sqlproc.engine.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sqlproc.engine.SqlQuery;
import org.sqlproc.engine.SqlRuntimeContext;
import org.sqlproc.engine.SqlRuntimeException;

/**
 * Holds the results of an explicit and an implicit mapping rules merging.
 * 
 * In the case of explicit mapping rule the grammar itself is defined in SqlMapping.g. In the case of implicit mapping
 * rule the grammar itself is defined in SqlStatement.g. The internal mapping rule is a dynamic one, and it's finalized
 * in the process of the final ANSI SQL query generation.
 * 
 * For the purpose of correct left join handling, the identities are identified. They are used to prevent the repeated
 * rows in the output result set. This is used for the associations (one-to-one, one-to-many and many-to-many).
 * 
 * The main runtime contracts are {@link #setQueryResultMapping(Class, Map, SqlQuery)} and
 * {@link SqlMappingResult#setQueryResultData(Object, Object[], Map, Map)}.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlMappingResult {

    /**
     * The internal slf4j logger.
     */
    final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * The crate for all input parameters and the context of processing.
     */
    private SqlProcessContext ctx;
    /**
     * All sub-elements based on ANTLR grammar defined in SqlMapping.g or SqlStatement.g. Every sub-element is one
     * Mapping item.
     */
    private Map<String, SqlMappingItem> mappings;
    /**
     * The collection of identities related to all output columns.
     */
    Map<String, SqlMappingIdentity> identities;
    /**
     * The list of identities indexes in the list of output values.
     */
    List<Integer> identitiesIndexes;
    /**
     * The main identity index in the list of output values. The main identity is related to the top level result class.
     */
    Integer mainIdentityIndex;

    /**
     * Creates a new instance. This instances is based on the merging of an explicit and an implicit mapping rules.
     * 
     * @param ctx
     *            the crate for all input parameters and the context of processing
     * @param mapping
     *            mapping rule based on SqlMapping.g or the empty one
     * @param outputMappings
     *            mapping rule items based on SqlStatement.g
     */
    SqlMappingResult(SqlProcessContext ctx, SqlMappingRule mapping, Map<String, SqlMappingItem> outputMappings) {
        this.ctx = ctx;
        mappings = new LinkedHashMap<String, SqlMappingItem>();
        identities = new LinkedHashMap<String, SqlMappingIdentity>();
        identitiesIndexes = new ArrayList<Integer>();
        merge(mapping, outputMappings);
    }

    /**
     * Returns the main identity index in the list of output values. The main identity is related to the top level
     * result class.
     * 
     * @return the main identity index in the list of output values
     */
    public Integer getMainIdentityIndex() {
        return mainIdentityIndex;
    }

    /**
     * Return the list of identities indexes in the list of output values
     * 
     * @return the list of identities indexes in the list of output values
     */
    public List<Integer> getIdentitiesIndexes() {
        return identitiesIndexes;
    }

    /**
     * Returns the public runtime context
     * 
     * @return the public runtime context
     */
    public SqlRuntimeContext getRuntimeContext() {
        return ctx;
    }

    /**
     * Adds a new mapping rule item in the merging process.
     * 
     * @param item
     *            a new mapping rule item
     */
    void addMapping(SqlMappingItem item) {
        mappings.put(item.getDbName(), item);
        if (item.getAttributes().isEmpty()) {
            String name = item.getName();
            if (!identities.containsKey(name)) {
                SqlMappingIdentity ident = new SqlMappingIdentity(item);
                identities.put(name, ident);
            }
        } else {
            for (SqlMappingAttribute attr : item.getAttributes()) {
                if (!identities.containsKey(attr.getFullName())) {
                    SqlMappingIdentity ident = new SqlMappingIdentity(item, attr);
                    identities.put(attr.getFullName(), ident);
                }
            }
        }
    }

    /**
     * Calculates all identities related information. They are used to prevent the repeated rows in the output result
     * set. This is used for the associations (one-to-one, one-to-many and many-to-many).
     */
    @SuppressWarnings("unused")
    void calculateIdentities() {
        int identityIndex = 0;
        for (SqlMappingItem item : mappings.values()) {
            if (item.isIdentity(ctx)) {
                identitiesIndexes.add(identityIndex);
                if (item.getName().equals(item.getFullName())) {
                    SqlMappingIdentity ident = identities.get(item.getName());
                    ident.addIdentityIndex(identityIndex, true);
                    ident.idenityDistance = 0;
                    mainIdentityIndex = identityIndex;
                } else {
                    int size = item.getAttributes().size();
                    int distance = 0;
                    for (int i = size - 1; i >= 0; --i, ++distance) {
                        SqlMappingAttribute attr = item.getAttributes().get(i);
                        SqlMappingIdentity ident = identities.get(attr.getFullName());
                        if (ident.identityIndexes != null && ident.idenityDistance < distance)
                            continue;
                        if (ident.identityIndexes != null && ident.idenityDistance == distance) {
                            ident.addIdentityIndex(identityIndex, false);
                            continue;
                        }
                        ident.addIdentityIndex(identityIndex, true);
                        ident.idenityDistance = distance;
                    }

                }
            }
            identityIndex++;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("===  calculateIdentities, identities=" + identities);
        }
        if (identitiesIndexes.isEmpty())
            return;
        for (SqlMappingItem item : mappings.values()) {
            if (!item.isIdentity(ctx)) {
                if (item.getAttributes().isEmpty()) {
                    SqlMappingIdentity ident = identities.get(item.getName());
                    ident.addIdentityIndex(mainIdentityIndex, true);
                } else {
                    int size = item.getAttributes().size();
                    for (int i = size - 1; i >= 0; --i) {
                        SqlMappingAttribute attr = item.getAttributes().get(i);
                        SqlMappingIdentity ident = identities.get(attr.getFullName());
                        if (ident.identityIndexes == null && i > 0) {
                            for (int j = i - 1; j >= 0; --j) {
                                SqlMappingAttribute attr2 = item.getAttributes().get(j);
                                SqlMappingIdentity ident2 = identities.get(attr2.getFullName());
                                if (ident2.identityIndexes != null) {
                                    ident.identityIndexes = ident2.identityIndexes;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        Map<List<Integer>, List<Integer>> parentIdentityIndexes = new HashMap<List<Integer>, List<Integer>>();
        List<Integer> mainIdentityIndexes = new ArrayList<Integer>();
        mainIdentityIndexes.add(mainIdentityIndex);
        for (Map.Entry<String, SqlMappingIdentity> entry : identities.entrySet()) {
            String fullName = entry.getKey();
            SqlMappingIdentity ident = entry.getValue();
            ident.allIdentityIndexes = new ArrayList<List<Integer>>();
            ident.allIdentityIndexes.add(mainIdentityIndexes);

            List<Integer> lastIdentityIndexes = null;
            int size = ident.item.getAttributes().size();
            for (int i = 0; i < size; i++) {
                SqlMappingAttribute attr = ident.item.getAttributes().get(i);
                SqlMappingIdentity ident2 = identities.get(attr.getFullName());
                if (ident2.identityIndexes == lastIdentityIndexes)
                    continue;
                boolean theSame = true;
                if (lastIdentityIndexes != null && ident2.identityIndexes.size() == lastIdentityIndexes.size()) {
                    for (int j = 0; j < lastIdentityIndexes.size(); j++) {
                        if (!ident2.identityIndexes.get(j).equals(lastIdentityIndexes.get(j))) {
                            theSame = false;
                            break;
                        }
                    }
                    if (!theSame)
                        ident.allIdentityIndexes.add(lastIdentityIndexes = ident2.identityIndexes);
                } else {
                    ident.allIdentityIndexes.add(lastIdentityIndexes = ident2.identityIndexes);
                }
            }
        }

        if (logger.isTraceEnabled()) {
            logger.trace("<<<  calculateIdentities, identities=" + identities + ", identitiesIndexes="
                    + identitiesIndexes + ", mainIdentityIndex=" + mainIdentityIndex);
        }
    }

    /**
     * Declares a scalar query results for all mapping rule items.
     * 
     * @param resultClass
     *            the class used for the return values
     * @param moreResultClasses
     *            more result classes used for the return values, like the collections classes or the collections items
     * @param query
     *            the SQL Engine query, an adapter or proxy to the internal JDBC or ORM staff
     * @throws org.sqlproc.engine.SqlRuntimeException
     *             in the case of any problem with output values preparation
     */
    public void setQueryResultMapping(Class<?> resultClass, Map<String, Class<?>> moreResultClasses, SqlQuery query)
            throws SqlRuntimeException {
        for (SqlMappingItem item : mappings.values()) {
            item.setQueryResultMapping(ctx, resultClass, moreResultClasses, query);
        }
    }

    /**
     * Fills the instance of the result class with output values from the SQL query execution.
     * 
     * @param resultInstance
     *            the instance of the result class
     * @param resultValues
     *            the query execution output values
     * @param ids
     *            the instances of all already used identities together with the related result instances based on
     *            identities indices
     * @param moreResultClasses
     *            more result classes used for the return values, like the collections classes or the collections items
     * @throws org.sqlproc.engine.SqlRuntimeException
     *             in the case of any problem with output values handling
     */
    public void setQueryResultData(Object resultInstance, Object[] resultValues, Map<String, Object> ids,
            Map<String, Class<?>> moreResultClasses) throws SqlRuntimeException {
        int i = 0;
        Map<String, Object> idsProcessed = new HashMap<String, Object>();

        for (SqlMappingItem item : mappings.values()) {
            item.setQueryResultData(ctx, resultInstance, i, resultValues, ids, idsProcessed, identities,
                    moreResultClasses);
            i++;
        }

        if (ids != null) {
            for (Map.Entry<String, Object> entry : idsProcessed.entrySet()) {
                ids.put(entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * Merge mapping rule for one META SQL query based on SqlMapping.g and SqlStatement.g. The external mapping rule has
     * the higher priority. The internal mapping rule holds the list of real output values.
     * 
     * @param mapping
     *            mapping rule based on SqlMapping.g or the empty one
     * @param outputMappings
     *            mapping rule items based on SqlStatement.g
     */
    private void merge(SqlMappingRule mapping, Map<String, SqlMappingItem> outputMappings) {
        if (outputMappings == null || outputMappings.isEmpty()) {
            for (SqlMappingItem mappingItem : mapping.getMappings().values()) {
                addMapping(mappingItem);
            }
        } else {
            for (SqlMappingItem mappingItem : outputMappings.values()) {
                if (mapping != null && mapping.getMappings().containsKey(mappingItem.getDbName())) {
                    addMapping(mapping.getMappings().get(mappingItem.getDbName()).merge(mappingItem));
                } else {
                    addMapping(mappingItem);
                }
            }
        }
        calculateIdentities();
    }

    /**
     * Construct the empty structure used for the instances of all already used identities together with the related
     * result instances based on identities indices.
     * 
     * @return the empty structure used for the instances of all already used identities together with the related
     *         result instances based on identities indices
     */
    public Map<String, Object> getIds() {
        if (getMainIdentityIndex() == null || getIdentitiesIndexes() == null)
            return null;
        Map<String, Object> ids = new HashMap<String, Object>();
        return ids;
    }

    /**
     * Devoted to function output value conversion.
     * 
     * @param result
     *            the JDBC output value
     * @return the output value after possible conversion
     */
    public Object getFunctionResultData(Map<String, Object> result) {
        for (Entry<String, Object> entry : result.entrySet()) {
            if (mappings.containsKey(entry.getKey())) {
                SqlMappingItem item = mappings.get(entry.getKey());
                SqlType sqlType = item.getSqlType();
                if (sqlType != null)
                    return sqlType.getResult(ctx, entry.getKey(), entry.getValue());
            }
        }
        return result.values().toArray()[0];
    }
}
