/*
 * Decompiled with CFR 0.152.
 */
package it.auties.buffer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.HexFormat;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.IntStream;

public class ByteBuffer {
    private static final String NULLABLE_MESSAGE = "Cannot create buffer from string: only non-null values are allowed inside named constructor %s(%s). Use %s(%s) instead if you want to accept nullable values";
    private static final HexFormat HEX_CODEC = HexFormat.of();
    private static final SecureRandom SECURE_RANDOM = new SecureRandom();
    private static final int DEFAULT_EXPAND_SIZE = 32;
    private static final int DEFAULT_EXPAND_MULTIPLIER = 4;
    private byte[] buffer;
    private int readerIndex;
    private int writerIndex;

    private ByteBuffer(byte[] buffer, boolean moveToEnd) {
        this.buffer = buffer;
        this.readerIndex = 0;
        this.writerIndex = moveToEnd ? buffer.length : 0;
    }

    public static ByteBuffer newBuffer() {
        return ByteBuffer.allocate(128);
    }

    public static ByteBuffer empty() {
        return ByteBuffer.of(0);
    }

    public static ByteBuffer of(int size) {
        return ByteBuffer.of(new byte[size]);
    }

    public static ByteBuffer of(byte ... input) {
        return new ByteBuffer(input, true);
    }

    public static ByteBuffer of(byte[] ... input) {
        return Arrays.stream(input).map(ByteBuffer::of).reduce(ByteBuffer.empty(), ByteBuffer::append);
    }

    public static ByteBuffer of(String input) {
        Objects.requireNonNull(input, NULLABLE_MESSAGE.formatted("of", "String", "ofNullable", "String"));
        return ByteBuffer.ofNullable(input);
    }

    public static ByteBuffer ofNullable(String input) {
        return ByteBuffer.of(input.getBytes(StandardCharsets.UTF_8));
    }

    public static ByteBuffer ofBase64(String input) {
        Objects.requireNonNull(input, NULLABLE_MESSAGE.formatted("ofBase64", "String", "ofNullable", "ofBase64"));
        return ByteBuffer.ofNullableBase64(input);
    }

    public static ByteBuffer ofNullableBase64(String input) {
        return ByteBuffer.of(Base64.getDecoder().decode(input));
    }

    public static ByteBuffer ofHex(String input) {
        Objects.requireNonNull(input, NULLABLE_MESSAGE.formatted("ofHex", "String", "ofNullable", "ofHex"));
        return ByteBuffer.ofNullableHex(input);
    }

    public static ByteBuffer ofNullableHex(String input) {
        return ByteBuffer.of(HEX_CODEC.parseHex(input));
    }

    public static ByteBuffer random(int length) {
        byte[] bytes = new byte[length];
        SECURE_RANDOM.nextBytes(bytes);
        return ByteBuffer.of(bytes);
    }

    public static ByteBuffer allocate(int length) {
        return new ByteBuffer(new byte[length], false);
    }

    public ByteBuffer prepend(ByteBuffer array) {
        Objects.requireNonNull(array, NULLABLE_MESSAGE.formatted("prepend", "ByteBuffer", "prependNullable", "ByteBuffer"));
        return this.prependNullable(array);
    }

    public ByteBuffer prependNullable(ByteBuffer array) {
        return array == null ? this.copy() : this.prependNullable(array.toByteArray());
    }

    public ByteBuffer prepend(byte ... array) {
        Objects.requireNonNull(array, NULLABLE_MESSAGE.formatted("prepend", "byte...", "prependNullable", "byte..."));
        return this.prependNullable(array);
    }

    public ByteBuffer prependNullable(byte ... array) {
        return ByteBuffer.of(array).appendNullable(this);
    }

    public ByteBuffer append(ByteBuffer array) {
        Objects.requireNonNull(array, NULLABLE_MESSAGE.formatted("append", "ByteBuffer", "appendNullable", "ByteBuffer"));
        return this.appendNullable(array);
    }

    public ByteBuffer appendNullable(ByteBuffer array) {
        return array == null ? this.copy() : this.appendNullable(array.toByteArray());
    }

    public ByteBuffer append(byte ... array) {
        Objects.requireNonNull(array, NULLABLE_MESSAGE.formatted("append", "byte...", "appendNullable", "byte..."));
        return this.appendNullable(array);
    }

    public ByteBuffer appendNullable(byte ... array) {
        if (array == null) {
            return this.copy();
        }
        byte[] result = Arrays.copyOf(this.toByteArray(), this.size() + array.length);
        System.arraycopy(array, 0, result, this.size(), array.length);
        return ByteBuffer.of(result);
    }

    public ByteBuffer cut(int end) {
        return this.slice(0, end);
    }

    public ByteBuffer slice(int start) {
        return this.slice(start, this.size());
    }

    public ByteBuffer slice(int start, int end) {
        return ByteBuffer.of(Arrays.copyOfRange(this.toByteArray(), start >= 0 ? start : this.size() + start, end >= 0 ? end : this.size() + end));
    }

    public ByteBuffer fill(char value) {
        return this.fill((int)value);
    }

    public ByteBuffer fill(Number value) {
        return this.fill(value, this.size());
    }

    public ByteBuffer fill(char value, int length) {
        return this.fill((int)value, length);
    }

    public ByteBuffer fill(Number value, int length) {
        byte[] result = new byte[this.size()];
        for (int index = 0; index < this.size(); ++index) {
            byte entry = this.buffer[index];
            result[index] = index < length && entry == 0 ? (Byte)value : entry;
        }
        return ByteBuffer.of(result);
    }

    public Optional<Byte> at(int index) {
        return this.size() <= index ? Optional.empty() : Optional.of(this.buffer[index]);
    }

    public OptionalInt indexOf(char entry) {
        return this.indexOf((int)entry);
    }

    public OptionalInt indexOf(Number entry) {
        return IntStream.range(0, this.size()).filter(index -> this.at(index).filter(entry::equals).isPresent()).findFirst();
    }

    public ByteBuffer assertSize(int size) {
        if (size != this.size()) {
            throw new AssertionError((Object)"Erroneous bytebuffer size: expected %s, got %s".formatted(size, this.size()));
        }
        return this;
    }

    public ByteBuffer writeByte(int input) {
        return this.writeByte(input, this.writerIndex);
    }

    public ByteBuffer writeByte(int input, int index) {
        return this.writeBytes(new byte[]{(byte)input}, index);
    }

    public ByteBuffer writeUnsignedByte(int input) {
        return this.writeUnsignedByte(input, this.writerIndex);
    }

    public ByteBuffer writeUnsignedByte(int input, int index) {
        return this.writeInt(Byte.toUnsignedInt((byte)input), index);
    }

    public ByteBuffer writeShort(int input) {
        return this.writeShort(input, this.writerIndex);
    }

    public ByteBuffer writeShort(int input, int index) {
        byte[] bytes = java.nio.ByteBuffer.allocate(2).putShort((short)input).array();
        this.writeBytes(bytes, index);
        return this;
    }

    public ByteBuffer writeUnsignedShort(short input) {
        return this.writeUnsignedShort(input, this.writerIndex);
    }

    public ByteBuffer writeUnsignedShort(short input, int index) {
        return this.writeInt(Short.toUnsignedInt(input), index);
    }

    public ByteBuffer writeInt(int input) {
        return this.writeInt(input, this.writerIndex);
    }

    public ByteBuffer writeInt(int input, int index) {
        byte[] bytes = java.nio.ByteBuffer.allocate(4).putInt(input).array();
        this.writeBytes(bytes, index);
        return this;
    }

    public ByteBuffer writeUnsignedInt(int input) {
        return this.writeUnsignedInt(input, this.writerIndex);
    }

    public ByteBuffer writeUnsignedInt(int input, int index) {
        return this.writeLong(Integer.toUnsignedLong(input), index);
    }

    public ByteBuffer writeLong(long input) {
        return this.writeLong(input, this.writerIndex);
    }

    public ByteBuffer writeLong(long input, int index) {
        byte[] bytes = java.nio.ByteBuffer.allocate(8).putLong(input).array();
        this.writeBytes(bytes, index);
        return this;
    }

    public ByteBuffer writeUnsignedLong(long input) {
        return this.writeUnsignedLong(input, this.writerIndex);
    }

    public ByteBuffer writeUnsignedLong(long input, int index) {
        return this.writeBigInt(this.toUnsignedBigInteger(input), index);
    }

    private BigInteger toUnsignedBigInteger(long i) {
        if (i >= 0L) {
            return BigInteger.valueOf(i);
        }
        int upper = (int)(i >>> 32);
        int lower = (int)i;
        return BigInteger.valueOf(Integer.toUnsignedLong(upper)).shiftLeft(32).add(BigInteger.valueOf(Integer.toUnsignedLong(lower)));
    }

    public ByteBuffer writeBigInt(BigInteger input) {
        return this.writeBigInt(input, this.writerIndex);
    }

    public ByteBuffer writeBigInt(BigInteger input, int index) {
        Objects.requireNonNull(input, NULLABLE_MESSAGE.formatted("writeBigInt", "BigInteger,int", "writeBigInt", "BigInteger,int"));
        return this.writeNullableBigInt(input, index);
    }

    public ByteBuffer writeNullableBigInt(BigInteger input) {
        return this.writeNullableBigInt(input, this.writerIndex);
    }

    public ByteBuffer writeNullableBigInt(BigInteger input, int index) {
        this.writeBytes(input.toByteArray(), index);
        return this;
    }

    public ByteBuffer writeBytes(byte[] bytes) {
        return this.writeBytes(bytes, this.writerIndex);
    }

    public ByteBuffer writeBytes(byte[] bytes, int index) {
        Objects.requireNonNull(bytes, NULLABLE_MESSAGE.formatted("writeBytes", "byte[],int", "writeNullableBytes", "byte[],int"));
        return this.writeNullableBytes(bytes, index);
    }

    public ByteBuffer writeNullableBytes(byte[] bytes, int index) {
        this.checkEOS(index, bytes.length);
        int inputIndex = 0;
        int bufferIndex = index;
        while (inputIndex < bytes.length) {
            this.buffer[bufferIndex] = bytes[inputIndex];
            ++inputIndex;
            ++bufferIndex;
        }
        this.step(bytes.length, false);
        return this;
    }

    public byte readByte() {
        return this.readByte(this.readerIndex);
    }

    public byte readByte(int index) {
        return this.readBytes(index, 1)[0];
    }

    public int readUnsignedByte() {
        return this.readUnsignedByte(this.readerIndex);
    }

    public int readUnsignedByte(int index) {
        return Byte.toUnsignedInt(this.readByte(index));
    }

    public short readShort() {
        return this.readShort(this.readerIndex);
    }

    public short readShort(int index) {
        return this.readBigInt(index, 2).shortValueExact();
    }

    public int readUnsignedShort() {
        return this.readUnsignedShort(this.readerIndex);
    }

    public int readUnsignedShort(int index) {
        return Short.toUnsignedInt(this.readShort(index));
    }

    public int readInt() {
        return this.readInt(this.readerIndex);
    }

    public int readInt(int index) {
        return this.readBigInt(index, 4).intValueExact();
    }

    public long readUnsignedInt() {
        return this.readUnsignedInt(this.readerIndex);
    }

    public long readUnsignedInt(int index) {
        return Integer.toUnsignedLong(this.readInt(index));
    }

    public long readLong() {
        return this.readLong(this.readerIndex);
    }

    public long readLong(int index) {
        return this.readBigInt(index, 8).longValueExact();
    }

    public BigInteger readUnsignedLong() {
        return this.readUnsignedLong(this.readerIndex);
    }

    public BigInteger readUnsignedLong(int index) {
        return new BigInteger(1, this.readBigInt(index, 8).toByteArray());
    }

    public BigInteger readBigInt(int length) {
        return this.readBigInt(this.readerIndex, length);
    }

    public BigInteger readBigInt(int index, int length) {
        return new BigInteger(this.readBytes(index, length));
    }

    public ByteBuffer readBuffer(int length) {
        return ByteBuffer.of(this.readBytes(this.readerIndex, length));
    }

    public ByteBuffer readBuffer(int index, int length) {
        return ByteBuffer.of(this.readBytes(index, length));
    }

    public byte[] readBytes(int length) {
        return this.readBytes(this.readerIndex, length);
    }

    public byte[] readBytes(int index, int length) {
        byte[] result = new byte[length];
        System.arraycopy(this.buffer, index, result, 0, length);
        this.step(length, true);
        return result;
    }

    private void checkEOS(int index, int delta) {
        if (index + delta < this.size()) {
            return;
        }
        int reservedSpace = Math.max(delta, 32) + 128;
        this.buffer = Arrays.copyOf(this.buffer, this.size() + reservedSpace);
    }

    private int step(int delta, boolean read) {
        int oldCounter;
        int n = oldCounter = read ? this.readerIndex : this.writerIndex;
        if (read) {
            this.readerIndex = oldCounter + delta;
        } else {
            this.writerIndex = oldCounter + delta;
        }
        return oldCounter;
    }

    public int size() {
        return this.buffer.length;
    }

    public int readableBytes() {
        return Math.max(this.size() - this.readerIndex, 0);
    }

    public int writableBytes() {
        return Math.max(this.size() - this.writerIndex, 0);
    }

    public boolean isReadable() {
        return this.size() - this.readerIndex > 0;
    }

    public boolean isWritable() {
        return this.size() - this.writerIndex > 0;
    }

    public byte[] toByteArray() {
        return Arrays.copyOfRange(this.buffer, 0, this.writerIndex);
    }

    public java.nio.ByteBuffer toNioBuffer() {
        return java.nio.ByteBuffer.wrap(this.toByteArray());
    }

    public ByteBuffer copy() {
        byte[] bytes = this.toByteArray();
        return ByteBuffer.of(Arrays.copyOf(bytes, bytes.length));
    }

    public ByteBuffer clear() {
        for (int index = 0; index < this.size(); ++index) {
            this.buffer[index] = 0;
        }
        this.readerIndex = 0;
        this.writerIndex = 0;
        return this;
    }

    public ByteBuffer remaining() {
        return ByteBuffer.of(this.readBytes(this.readerIndex, this.size() - this.readerIndex));
    }

    public String toString() {
        return new String(this.toByteArray(), StandardCharsets.UTF_8);
    }

    public ByteArrayOutputStream toOutputStream() throws IOException {
        ByteArrayOutputStream stream = new ByteArrayOutputStream(this.size());
        stream.write(this.toByteArray());
        return stream;
    }

    public String toHex() {
        return HEX_CODEC.formatHex(this.toByteArray());
    }

    public String toBase64() {
        return Base64.getEncoder().encodeToString(this.toByteArray());
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public boolean equals(Object other) {
        if (this == other) return true;
        if (!(other instanceof ByteBuffer)) return false;
        ByteBuffer that = (ByteBuffer)other;
        if (!Arrays.equals(this.toByteArray(), that.toByteArray())) return false;
        return true;
    }

    public boolean contentEquals(Object other) {
        byte[] thoseBytes;
        java.nio.ByteBuffer thatNioBuffer;
        return this.equals(other) || other instanceof java.nio.ByteBuffer && (thatNioBuffer = (java.nio.ByteBuffer)other).equals(this.toNioBuffer()) || other instanceof byte[] && Arrays.equals(thoseBytes = (byte[])other, this.toByteArray());
    }
}

