/*
 * Java-systemd implementation
 * Copyright (c) 2016 Markus Enax
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of either the GNU Lesser General Public License Version 2 or the
 * Academic Free Licence Version 3.0.
 *
 * Full licence texts are included in the COPYING file with this program.
 */

package de.thjom.java.systemd;

import java.math.BigInteger;
import java.util.List;
import java.util.Objects;

import org.freedesktop.dbus.connections.impl.DBusConnection;
import org.freedesktop.dbus.exceptions.DBusException;
import org.freedesktop.dbus.types.UInt64;
import org.freedesktop.dbus.types.Variant;

import de.thjom.java.systemd.interfaces.PropertyInterface;

public class Properties extends InterfaceAdapter {

    public static final String SERVICE_NAME = "org.freedesktop.DBus.Properties";

    private final String serviceName;

    private Properties(final DBusConnection dbus, final PropertyInterface iface, final String serviceName) {
        super(dbus, iface);

        this.serviceName = Objects.requireNonNull(serviceName);
    }

    static Properties create(final DBusConnection dbus, final String objectPath, final String serviceName) throws DBusException {
        PropertyInterface iface = dbus.getRemoteObject(Systemd.SERVICE_NAME, objectPath, PropertyInterface.class);

        return new Properties(dbus, iface, serviceName);
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }

        if (obj == null) {
            return false;
        }

        if (this.getClass() == obj.getClass()) {
            Properties other = (Properties) obj;

            return super.equals(obj) && serviceName.equals(other.serviceName);
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), serviceName);
    }

    @Override
    public PropertyInterface getInterface() {
        return (PropertyInterface) super.getInterface();
    }

    /**
     * Returns this interface adapter (unlike the implementations in the other
     * interface adapters).
     */
    @Override
    public Properties getProperties() {
        return this;
    }

    public Variant<?> getVariant(final String propertyName) {
        return getInterface().getProperty(serviceName, propertyName);
    }

    public boolean getBoolean(final String propertyName) {
        return (Boolean) getVariant(propertyName).getValue();
    }

    public byte getByte(final String propertyName) {
        return (Byte) getVariant(propertyName).getValue();
    }

    public short getShort(final String propertyName) {
        Number value = (Number) getVariant(propertyName).getValue();

        return value.shortValue();
    }

    public int getInteger(final String propertyName) {
        Number value = (Number) getVariant(propertyName).getValue();

        return value.intValue();
    }

    public long getLong(final String propertyName) {
        Number value = (Number) getVariant(propertyName).getValue();

        return value.longValue();
    }

    public BigInteger getBigInteger(final String propertyName) {
        UInt64 value = (UInt64) getVariant(propertyName).getValue();

        return value.value();
    }

    public double getDouble(final String propertyName) {
        Number value = (Number) getVariant(propertyName).getValue();

        return value.doubleValue();
    }

    public String getString(final String propertyName) {
        return String.valueOf(getVariant(propertyName).getValue());
    }

    @SuppressWarnings("unchecked")
    public <T> List<T> getList(final String propertyName) {
        return (List<T>) getVariant(propertyName).getValue();
    }

}
