/*
 * Decompiled with CFR 0.152.
 */
package com.simiacryptus.util.text;

import com.simiacryptus.util.text.CharTrie;
import com.simiacryptus.util.text.TrieNode;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.TreeMap;
import java.util.function.ToDoubleFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

public class TextAnalysis {
    public static final double DEFAULT_THRESHOLD = Math.log(15.0);
    private final CharTrie inner;
    private PrintStream verbose = null;

    TextAnalysis(CharTrie inner) {
        this.inner = inner;
    }

    public List<String> keywords(String source) {
        Map<String, Long> wordCounts = this.splitChars(source, DEFAULT_THRESHOLD).stream().collect(Collectors.groupingBy(x -> x, Collectors.counting()));
        wordCounts = this.aggregateKeywords(wordCounts);
        return wordCounts.entrySet().stream().filter(x -> (Long)x.getValue() > 1L).sorted(Comparator.comparing(x -> -this.entropy((String)x.getKey()) * Math.pow(((Long)x.getValue()).longValue(), 0.3))).map(e -> {
            if (this.isVerbose()) {
                this.verbose.println(String.format("KEYWORD: \"%s\" - %s * %.3f / %s", e.getKey(), e.getValue(), this.entropy((String)e.getKey()), ((String)e.getKey()).length()));
            }
            return (String)e.getKey();
        }).collect(Collectors.toList());
    }

    private Map<String, Long> aggregateKeywords(Map<String, Long> wordCounts) {
        HashMap<String, Long> accumulator = new HashMap<String, Long>();
        wordCounts.forEach((key, count) -> {
            boolean added = false;
            for (Map.Entry e : accumulator.entrySet()) {
                String combine = TextAnalysis.combine(key, (String)e.getKey(), 4);
                if (null == combine) continue;
                accumulator.put(combine, (Long)e.getValue() + count);
                accumulator.remove(e.getKey());
                added = true;
                break;
            }
            if (!added) {
                accumulator.put((String)key, (Long)count);
            }
        });
        if (wordCounts.size() > accumulator.size()) {
            return this.aggregateKeywords(accumulator);
        }
        return accumulator;
    }

    public static String combine(String left, String right, int minOverlap) {
        if (left.length() < minOverlap) {
            return null;
        }
        if (right.length() < minOverlap) {
            return null;
        }
        int bestOffset = Integer.MAX_VALUE;
        for (int offset = minOverlap - left.length(); offset < right.length() - minOverlap; ++offset) {
            boolean match = true;
            for (int posLeft = Math.max(0, -offset); posLeft < Math.min(left.length(), right.length() - offset); ++posLeft) {
                if (left.charAt(posLeft) == right.charAt(posLeft + offset)) continue;
                match = false;
                break;
            }
            if (!match || Math.abs(bestOffset) <= Math.abs(offset)) continue;
            bestOffset = offset;
        }
        if (bestOffset < Integer.MAX_VALUE) {
            String combined = left;
            if (bestOffset > 0) {
                combined = right.substring(0, bestOffset) + combined;
            }
            if (left.length() + bestOffset < right.length()) {
                combined = combined + right.substring(left.length() + bestOffset);
            }
            return combined;
        }
        return null;
    }

    public double spelling(String source) {
        assert (source.startsWith("|"));
        assert (source.endsWith("|"));
        WordSpelling original = new WordSpelling(source);
        WordSpelling corrected = IntStream.range(0, 1).mapToObj(i -> this.buildCorrection(original)).min(Comparator.comparingDouble(x -> x.sum)).get();
        return corrected.sum;
    }

    private WordSpelling buildCorrection(WordSpelling wordSpelling) {
        int timesWithoutImprovement = 0;
        int maxCorrections = 10;
        int trials = 10;
        if (null != this.verbose) {
            this.verbose.println(String.format("START: \"%s\"\t%.5f", wordSpelling.text, wordSpelling.sum));
        }
        while (timesWithoutImprovement++ < 100) {
            WordSpelling _wordSpelling = wordSpelling;
            ToDoubleFunction<WordSpelling> fitness = mutant -> mutant.sum * 1.0 / (double)((WordSpelling)mutant).text.length();
            WordSpelling mutant2 = wordSpelling.mutate().filter(x -> {
                if (!((WordSpelling)x).text.startsWith("|")) {
                    return false;
                }
                return ((WordSpelling)x).text.endsWith("|");
            }).min(Comparator.comparingDouble(fitness::applyAsDouble)).get();
            if (fitness.applyAsDouble(mutant2) < fitness.applyAsDouble(wordSpelling)) {
                if (null != this.verbose) {
                    this.verbose.println(String.format("IMPROVEMENT: \"%s\"\t%.5f", mutant2.text, mutant2.sum));
                }
                wordSpelling = mutant2;
                timesWithoutImprovement = 0;
                if (maxCorrections-- <= 0) break;
            }
            if (!this.inner.contains(wordSpelling.text)) continue;
            if (null == this.verbose) break;
            this.verbose.println(String.format("WORD: \"%s\"\t%.5f", mutant2.text, mutant2.sum));
            break;
        }
        return wordSpelling;
    }

    public List<String> splitMatches(String text, int minSize) {
        TrieNode node = this.inner.root();
        ArrayList<String> matches = new ArrayList<String>();
        String accumulator = "";
        for (int i = 0; i < text.length(); ++i) {
            short prevDepth = node.getDepth();
            TrieNode prevNode = node;
            if (null == (node = node.getContinuation(text.charAt(i)))) {
                node = this.inner.root();
            }
            if (!accumulator.isEmpty() && (node.getDepth() < prevDepth || prevNode.hasChildren() && node.getDepth() == prevDepth)) {
                if (accumulator.length() > minSize) {
                    matches.add(accumulator);
                    node = this.inner.root().getChild(text.charAt(i)).orElse(this.inner.root());
                }
                accumulator = "";
                continue;
            }
            if (!accumulator.isEmpty()) {
                accumulator = accumulator + text.charAt(i);
                continue;
            }
            if (!accumulator.isEmpty() || node.getDepth() <= prevDepth) continue;
            accumulator = node.getString();
        }
        ArrayList<String> tokenization = new ArrayList<String>();
        for (String match : matches) {
            int index = text.indexOf(match);
            assert (index >= 0);
            if (index > 0) {
                tokenization.add(text.substring(0, index));
            }
            tokenization.add(text.substring(index, index + match.length()));
            text = text.substring(index + match.length());
        }
        tokenization.add(text);
        return tokenization;
    }

    public List<String> splitChars(String source, double threshold) {
        ArrayList<String> output = new ArrayList<String>();
        int wordStart = 0;
        double aposterioriNatsPrev = 0.0;
        boolean isIncreasing = false;
        double prevLink = 0.0;
        for (int i = 1; i < source.length(); ++i) {
            String word;
            String priorText = source.substring(0, i);
            TrieNode priorNode = this.getMaxentPrior(priorText);
            double aprioriNats = TextAnalysis.entropy(priorNode, priorNode.getParent());
            String followingText = source.substring(i - 1, source.length());
            TrieNode followingNode = this.getMaxentPost(followingText);
            TrieNode godparent = followingNode.godparent();
            double aposterioriNats = TextAnalysis.entropy(followingNode, godparent);
            double linkNats = aprioriNats + aposterioriNatsPrev;
            if (this.isVerbose()) {
                this.verbose.println(String.format("%10s\t%10s\t%s", '\"' + priorNode.getString().replaceAll("\n", "\\n") + '\"', '\"' + followingNode.getString().replaceAll("\n", "\\n") + '\"', Arrays.asList(aprioriNats, aposterioriNats, linkNats).stream().map(x -> String.format("%.4f", x)).collect(Collectors.joining("\t"))));
            }
            String string = word = i < 2 ? "" : source.substring(wordStart, i - 2);
            if (isIncreasing && linkNats < prevLink && prevLink > threshold && word.length() > 2) {
                wordStart = i - 2;
                output.add(word);
                if (this.isVerbose()) {
                    this.verbose.println(String.format("Recognized token \"%s\"", word));
                }
                prevLink = linkNats;
                aposterioriNatsPrev = aposterioriNats;
                isIncreasing = false;
                continue;
            }
            if (linkNats > prevLink) {
                isIncreasing = true;
            }
            prevLink = linkNats;
            aposterioriNatsPrev = aposterioriNats;
        }
        return output;
    }

    private TrieNode getMaxentPost(String followingText) {
        TrieNode godparent2;
        String followingText2;
        TrieNode followingNode2;
        double aposterioriNats2;
        TrieNode followingNode = this.inner.traverse(followingText);
        TrieNode godparent1 = followingNode.godparent();
        double aposterioriNats1 = TextAnalysis.entropy(followingNode, godparent1);
        while (followingText.length() > 1 && (aposterioriNats2 = TextAnalysis.entropy(followingNode2 = this.inner.traverse(followingText2 = followingText.substring(0, followingText.length() - 1)), godparent2 = followingNode2.godparent())) < aposterioriNats1) {
            aposterioriNats1 = aposterioriNats2;
            followingNode = followingNode2;
            followingText = followingText2;
        }
        return followingNode;
    }

    private TrieNode getMaxentPrior(String priorText) {
        String priorText2;
        TrieNode priorNode2;
        double aprioriNats2;
        TrieNode priorNode = this.inner.matchEnd(priorText);
        double aprioriNats1 = TextAnalysis.entropy(priorNode, priorNode.getParent());
        while (priorText.length() > 1 && (aprioriNats2 = TextAnalysis.entropy(priorNode2 = this.inner.matchEnd(priorText2 = priorText.substring(1)), priorNode2.getParent())) < aprioriNats1) {
            aprioriNats1 = aprioriNats2;
            priorText = priorText2;
            priorNode = priorNode2;
        }
        return priorNode;
    }

    private double getJointNats(TrieNode priorNode, TrieNode followingNode) {
        Map<Character, Long> code = this.getJointExpectation(priorNode, followingNode);
        double sumOfProduct = code.values().stream().mapToDouble(x -> x.longValue()).sum();
        double product = followingNode.getCursorCount() * priorNode.getCursorCount();
        return -Math.log(product / sumOfProduct);
    }

    private Map<Character, Long> getJointExpectation(TrieNode priorNode, TrieNode followingNode) {
        TrieNode priorParent = priorNode.getParent();
        TreeMap<Character, ? extends TrieNode> childrenMap = null == priorParent ? this.inner.root().getChildrenMap() : priorParent.getChildrenMap();
        String followingString = followingNode.getString();
        String postContext = followingString.isEmpty() ? "" : followingString.substring(1);
        return childrenMap.keySet().stream().collect(Collectors.toMap(x -> x, token -> {
            TrieNode altFollowing = this.inner.traverse(token + postContext);
            long a = altFollowing.getString().equals(token + postContext) ? altFollowing.getCursorCount() : 0L;
            TrieNode parent = priorParent;
            long b = ((TrieNode)childrenMap.get(token)).getCursorCount();
            return a * b;
        }));
    }

    private static double entropy(TrieNode tokenNode, TrieNode contextNode) {
        return -0.0 + (null == contextNode ? Double.POSITIVE_INFINITY : -Math.log((double)tokenNode.getCursorCount() * 1.0 / (double)contextNode.getCursorCount()));
    }

    public double entropy(String source) {
        double output = 0.0;
        for (int i = 1; i < source.length(); ++i) {
            TrieNode node = this.inner.matchEnd(source.substring(0, i));
            Optional<? extends TrieNode> child = node.getChild(source.charAt(i));
            while (!child.isPresent()) {
                output += Math.log(1.0 / (double)node.getCursorCount());
                node = node.godparent();
                child = node.getChild(source.charAt(i));
            }
            output += Math.log((double)child.get().getCursorCount() * 1.0 / (double)node.getCursorCount());
        }
        return -output / Math.log(2.0);
    }

    public boolean isVerbose() {
        return null != this.verbose;
    }

    public TextAnalysis setVerbose(PrintStream verbose) {
        this.verbose = verbose;
        return this;
    }

    public List<String> splitChars(String text) {
        return this.splitChars(text, DEFAULT_THRESHOLD);
    }

    public class WordSpelling {
        private final double[] linkNatsArray;
        private final List<TrieNode> leftNodes;
        private final List<TrieNode> rightNodes;
        private final String text;
        double sum = 0.0;
        private final Random random = new Random();

        public WordSpelling(String source) {
            this.text = source;
            this.linkNatsArray = new double[source.length()];
            this.leftNodes = new ArrayList<TrieNode>(source.length());
            this.rightNodes = new ArrayList<TrieNode>(source.length());
            TrieNode priorNode = TextAnalysis.this.inner.root();
            double aposterioriNatsPrev = 0.0;
            for (int i = 1; i <= source.length(); ++i) {
                double jointNats;
                priorNode = priorNode.getContinuation(source.charAt(i - 1));
                double aprioriNats = TextAnalysis.entropy(priorNode, priorNode.getParent());
                TrieNode followingNode = TextAnalysis.this.inner.traverse(source.substring(i - 1, source.length()));
                this.leftNodes.add(priorNode);
                this.rightNodes.add(followingNode);
                double aposterioriNats = TextAnalysis.entropy(followingNode, followingNode.godparent());
                Map code = TextAnalysis.this.getJointExpectation(priorNode, followingNode);
                double sumOfProduct = code.values().stream().mapToDouble(x1 -> x1.longValue()).sum();
                double product = followingNode.getCursorCount() * priorNode.getCursorCount();
                this.linkNatsArray[i - 1] = jointNats = -Math.log(product / sumOfProduct);
                this.sum += jointNats;
                aposterioriNatsPrev = aposterioriNats;
            }
            double sumLinkNats = Arrays.stream(this.linkNatsArray).sum();
            int i = 0;
            while (i < this.linkNatsArray.length) {
                int n = i++;
                this.linkNatsArray[n] = this.linkNatsArray[n] / sumLinkNats;
            }
        }

        public Stream<WordSpelling> mutate() {
            return IntStream.range(0, this.linkNatsArray.length).mapToObj(x -> x).sorted(Comparator.comparingDouble(i1 -> this.linkNatsArray[i1])).flatMap(i -> this.mutateAt((int)i));
        }

        private Stream<WordSpelling> mutateAt(int pos) {
            return IntStream.range(0, 6).mapToObj(x -> x).flatMap(fate -> {
                if (fate == 0) {
                    return this.mutateDeletion(pos);
                }
                if (fate == 1) {
                    return this.mutateSubstitution(pos);
                }
                if (fate == 2) {
                    return this.mutateAddLeft(pos);
                }
                if (fate == 3) {
                    return this.mutateAddRight(pos);
                }
                if (fate == 4) {
                    return this.mutateSwapLeft(pos);
                }
                if (fate == 5) {
                    return this.mutateSwapRight(pos);
                }
                return Stream.empty();
            });
        }

        private Stream<WordSpelling> mutateSwapRight(int pos) {
            if (this.text.length() - 1 <= pos) {
                return Stream.empty();
            }
            char[] charArray = this.text.toCharArray();
            char temp = charArray[pos + 1];
            charArray[pos + 1] = charArray[pos];
            charArray[pos] = temp;
            return Stream.of(new WordSpelling(new String(charArray)));
        }

        private Stream<WordSpelling> mutateSwapLeft(int pos) {
            if (0 >= pos) {
                return Stream.empty();
            }
            char[] charArray = this.text.toCharArray();
            char temp = charArray[pos - 1];
            charArray[pos - 1] = charArray[pos];
            charArray[pos] = temp;
            return Stream.of(new WordSpelling(new String(charArray)));
        }

        private Stream<WordSpelling> mutateAddRight(int pos) {
            Stream<Character> newCharStream = this.pick(TextAnalysis.this.getJointExpectation(this.text.length() - 1 <= pos ? TextAnalysis.this.inner.root() : this.leftNodes.get(pos + 1), this.rightNodes.get(pos)));
            return newCharStream.map(newChar -> new WordSpelling(this.text.substring(0, pos) + newChar + this.text.substring(pos)));
        }

        private Stream<WordSpelling> mutateAddLeft(int pos) {
            Stream<Character> newCharStream = this.pick(TextAnalysis.this.getJointExpectation(this.leftNodes.get(pos), 0 >= pos ? TextAnalysis.this.inner.root() : this.rightNodes.get(pos - 1)));
            return newCharStream.map(newChar -> new WordSpelling(this.text.substring(0, pos) + newChar + this.text.substring(pos)));
        }

        private Stream<WordSpelling> mutateSubstitution(int pos) {
            Stream<Character> newCharStream = this.pick(TextAnalysis.this.getJointExpectation(this.leftNodes.get(pos), this.rightNodes.get(pos)));
            return newCharStream.map(newChar -> {
                char[] charArray = this.text.toCharArray();
                charArray[pos] = newChar.charValue();
                return new WordSpelling(new String(charArray));
            });
        }

        private Stream<Character> pick(Map<Character, Long> weights) {
            return weights.entrySet().stream().sorted(Comparator.comparingLong(e -> (Long)e.getValue())).map(e -> (Character)e.getKey());
        }

        private Stream<WordSpelling> mutateDeletion(int pos) {
            return Stream.of(new WordSpelling(this.text.substring(0, pos) + this.text.substring(pos + 1)));
        }
    }
}

