/*
 * Decompiled with CFR 0.152.
 */
package io.pdfapi.client;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.pdfapi.client.ConversionRequest;
import io.pdfapi.client.PdfApiClientConfig;
import io.pdfapi.client.PdfApiClientException;
import io.pdfapi.client.http.HttpClient;
import io.pdfapi.client.http.HttpResponse;
import io.pdfapi.client.model.ConversionProperties;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfApiClient
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(PdfApiClient.class);
    private static final String HEADER_API_KEY = "Api-Key";
    private static final long INITIAL_POLLING_DELAY_MS = 500L;
    private static final long MAX_POLLING_DELAY_MS = 5000L;
    private static final float BACKOFF_MULTIPLIER = 1.5f;
    private static final String PATH_CONVERSIONS = "/api/conversions";
    private static final String PATH_ASSETS = "/assets";
    private static final String PATH_CONVERT = "/convert";
    private final HttpClient httpClient;
    private final ObjectMapper objectMapper;
    private final String baseUrl;
    private final String apiKey;

    protected PdfApiClient(PdfApiClientConfig config, HttpClient httpClient) {
        this.baseUrl = config.getBaseUrl();
        this.apiKey = config.getApiKey();
        this.httpClient = httpClient;
        this.objectMapper = new ObjectMapper();
    }

    public CompletableFuture<InputStream> convert(ConversionRequest request) {
        logger.info("Starting PDF conversion");
        return this.initializeConversion(request.getProperties()).thenCompose(conversionId -> {
            logger.debug("Conversion initialized with ID: {}", conversionId);
            return ((CompletableFuture)this.uploadAssetsInParallel((String)conversionId, request.getAssets()).thenCompose(v -> {
                logger.debug("Assets uploaded for conversion {}", conversionId);
                return this.performConversion((String)conversionId, request.getHtmlContent());
            })).thenCompose(resultLocation -> {
                logger.debug("Starting to wait for conversion result {}", conversionId);
                return this.waitForResult((String)conversionId, (String)resultLocation);
            });
        });
    }

    CompletableFuture<Void> convert(ConversionRequest request, OutputStream output) {
        return this.convert(request).thenAccept(pdfStream -> {
            try (InputStream is = pdfStream;){
                is.transferTo(output);
            }
            catch (Exception e) {
                throw new PdfApiClientException("Failed to write PDF content", e);
            }
        });
    }

    void convertSync(ConversionRequest request, OutputStream output) {
        this.convert(request, output).join();
    }

    private CompletableFuture<String> initializeConversion(ConversionProperties properties) {
        try {
            String json = this.objectMapper.writeValueAsString((Object)properties);
            logger.debug("Initializing conversion with properties: {}", (Object)json);
            return this.httpClient.post(this.baseUrl + PATH_CONVERSIONS, this.getHeaders(), json).thenApply(response -> this.parseJsonResponse((HttpResponse)response, "id"));
        }
        catch (JsonProcessingException e) {
            logger.error("Failed to serialize conversion properties", (Throwable)e);
            return CompletableFuture.failedFuture(new PdfApiClientException("Failed to serialize conversion properties", e));
        }
    }

    private CompletableFuture<Void> uploadAssetsInParallel(String conversionId, List<ConversionRequest.AssetInput> assets) {
        logger.debug("Uploading {} assets for conversion {}", (Object)assets.size(), (Object)conversionId);
        List uploads = assets.stream().map(asset -> this.attachAsset(conversionId, asset.getContent(), asset.getFileName())).collect(Collectors.toList());
        return CompletableFuture.allOf((CompletableFuture[])uploads.toArray(CompletableFuture[]::new));
    }

    private CompletableFuture<Void> attachAsset(String conversionId, InputStream assetStream, String fileName) {
        logger.debug("Attaching asset {} to conversion {}", (Object)fileName, (Object)conversionId);
        return this.httpClient.post(this.baseUrl + "/api/conversions/" + conversionId + PATH_ASSETS, this.getHeaders(), fileName, assetStream, "application/octet-stream", "asset").thenApply(response -> {
            this.handleResponse((HttpResponse)response);
            return null;
        });
    }

    private CompletableFuture<String> performConversion(String conversionId, InputStream htmlContent) {
        logger.debug("Starting conversion for ID: {}", (Object)conversionId);
        return this.httpClient.post(this.baseUrl + "/api/conversions/" + conversionId + PATH_CONVERT, this.getHeaders(), "index.html", htmlContent, "text/html", "index").thenApply(response -> {
            HttpResponse r = this.handleResponse((HttpResponse)response);
            logger.debug("Conversion started successfully for ID: {}", (Object)conversionId);
            return r.getLocationHeader().orElseThrow(() -> new PdfApiClientException("Result location not returned during conversion"));
        });
    }

    private CompletableFuture<InputStream> waitForResult(String conversionId, String resultLocation) {
        return this.waitForResultWithBackoff(conversionId, resultLocation, 500L);
    }

    private CompletableFuture<InputStream> waitForResultWithBackoff(String conversionId, String resultLocation, long currentDelay) {
        logger.trace("Checking conversion status for {} with delay {}ms", (Object)conversionId, (Object)currentDelay);
        return this.getConversionResult(resultLocation).thenCompose(result -> {
            if (result == null) {
                logger.trace("Conversion {} still in progress, next check in {}ms", (Object)conversionId, (Object)Math.min((long)((float)currentDelay * 1.5f), 5000L));
                CompletableFuture delay = new CompletableFuture();
                CompletableFuture.delayedExecutor(currentDelay, TimeUnit.MILLISECONDS).execute(() -> delay.complete(null));
                return delay.thenCompose(v -> this.waitForResultWithBackoff(conversionId, resultLocation, Math.min((long)((float)currentDelay * 1.5f), 5000L)));
            }
            logger.info("Conversion {} completed successfully", (Object)conversionId);
            return CompletableFuture.completedFuture(result);
        });
    }

    private CompletableFuture<InputStream> getConversionResult(String resultLocation) {
        return this.httpClient.get(resultLocation, this.getHeaders()).thenApply(response -> {
            if (response.getStatusCode() == 204) {
                return null;
            }
            return this.handleResponse((HttpResponse)response).getBodyAsStream();
        });
    }

    private String parseJsonResponse(HttpResponse response, String field) {
        this.handleResponse(response);
        try {
            return this.objectMapper.readTree(response.getBodyAsStream()).get(field).asText();
        }
        catch (IOException e) {
            throw new PdfApiClientException("Failed to parse JSON response", e);
        }
    }

    private HttpResponse handleResponse(HttpResponse response) {
        if (response.getStatusCode() != 200 && response.getStatusCode() != 201 && response.getStatusCode() != 204) {
            logger.error("Request failed with status {}", (Object)response.getStatusCode());
            throw new PdfApiClientException("Request failed with status " + response.getStatusCode());
        }
        return response;
    }

    private Map<String, String> getHeaders() {
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put(HEADER_API_KEY, this.apiKey);
        return Collections.unmodifiableMap(headers);
    }

    @Override
    public void close() {
        logger.debug("Closing PDF API client");
        this.httpClient.close();
    }
}

