package fi.evolver.script.app;

import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;

import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Git {

	public static void install() {
		try (Step step = Step.start("Install git")) {
			Apt.install("git");
		}
	}

	public static void configureUser(String name, String email) {
		try (Step step = Step.start("Git: Configure user")) {
			if (name == null || name.isBlank())
				step.fail("Git user name is empty");
			if (email == null || email.isBlank())
				step.fail("Git user email is empty");

			Shell.user("git", "config", "--global", "user.name", name);
			Shell.user("git", "config", "--global", "user.email", email);
		}
	}

	public static void clone(String repository, Path target) {
		try (Step step = Step.start("Git clone %s".formatted(repository.replaceFirst(".*/", "").replaceFirst("\\.git$", "")))) {
			install();
			if (target.toFile().exists()) {
				step.skip("Clone target already exists");
				return;
			}
			Shell.user("git", "clone", repository, target.toAbsolutePath().toString());
		}
	}

	public static void addAliases(Map<String, String> aliases) {
		try (Step step = Step.start("Git: Add aliases: " + aliases.keySet())) {
			aliases.forEach((alias, definition) -> {
				Shell.user("git", "config", "--global", "alias." + alias, definition);
			});
		}
	}

	public static Optional<String> getUserName() {
		return getGlobalConfig("user.name");
	}

	public static Optional<String> getEmail() {
		return getGlobalConfig("user.email");
	}

	/**
	 * @param name the name of the global configuration option
	 * @return the value of <code>name</code> or empty if <code>name</code> is not configured
	 */
	public static Optional<String> getGlobalConfig(String name) {
		Shell.Result result = Shell.Command.user("git", "config", "--global", name)
				.failOnError(false)
				.run();

		if (result.success()) {
			return Optional.of(result.stdout().trim());
		} else {
			return Optional.empty();
		}
	}


}
