package fi.evolver.script;

import static org.junit.jupiter.api.Assertions.assertEquals;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class EnvironmentTest {
    
	@Test
	public void escapeValue() {
		assertEquals("''\\'''", Environment.escapeValue("'"));
		assertEquals("'\"'", Environment.escapeValue("\""));
		assertEquals("' '", Environment.escapeValue(" "));
		assertEquals("''", Environment.escapeValue(""));
	}

	@Test
	public void unescapeValue() {
		assertEquals("'", Environment.unescapeValue("''\\'''"));
		assertEquals("\"", Environment.unescapeValue("\"\"\""));
		assertEquals("\"", Environment.unescapeValue("\\\""));
		assertEquals("$", Environment.unescapeValue("$"));
		assertEquals(" ", Environment.unescapeValue(" "));
		assertEquals("", Environment.unescapeValue(""));
	}

    @Test
    public void escapeAndUnescapeAllValidCharacters() {
        StringBuilder allChars = new StringBuilder();

        // Iterate through all valid Unicode code points
        for (int codePoint = 0; codePoint <= Character.MAX_CODE_POINT; codePoint++) {
            // Skip invalid code points (surrogates, non-characters, unassigned)
            if (Character.isValidCodePoint(codePoint) &&
                !Character.isSurrogate((char) codePoint)) {
                allChars.appendCodePoint(codePoint);
            }
        }

        String input = allChars.toString();
        String escaped = Environment.escapeValue(input);
        String unescaped = Environment.unescapeValue(escaped);

        assertEquals(input, unescaped);
    }

    @Test
    public void writeAndReadVariables(@TempDir Path tempDir) throws IOException {
        Path temp = tempDir.resolve("temp-variables");
        Collection<Environment.Variable> expectedVariables = List.of(
                new Environment.Variable("name1", "value1", null),
                new Environment.Variable("name2", "value2", "comment2"),
                new Environment.Variable("name3", "value3-with-long-value                  ", "comment3"),
                new Environment.Variable("name4", "value4", null)
        );

        Environment.writeVariables(temp, expectedVariables);

        Map<String, Environment.Variable> actualVariables = Environment.readVariables(temp);
        for (Environment.Variable expectedVariable : expectedVariables) {
            assertEquals(expectedVariable, actualVariables.get(expectedVariable.name()));
        }
    }
}
