/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.api.data.schema.UnsupportedTypeException;
import io.cdap.plugin.DriverCleanup;
import io.cdap.plugin.JDBCDriverShim;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import javax.annotation.Nullable;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DBUtils {
    private static final Logger LOG = LoggerFactory.getLogger(DBUtils.class);
    public static final String OVERRIDE_SCHEMA = "io.cdap.hydrator.db.override.schema";
    public static final String CONNECTION_ARGUMENTS = "io.cdap.hydrator.db.connection.arguments";

    public static void cleanup(Class<? extends Driver> driverClass) {
        ClassLoader pluginClassLoader = driverClass.getClassLoader();
        if (pluginClassLoader == null) {
            LOG.warn("PluginClassLoader is null. Cleanup not necessary.");
            return;
        }
        DBUtils.shutDownMySQLAbandonedConnectionCleanupThread(pluginClassLoader);
        DBUtils.unregisterOracleMBean(pluginClassLoader);
    }

    public static DriverCleanup ensureJDBCDriverIsAvailable(Class<? extends Driver> jdbcDriverClass, String connectionString, String jdbcPluginType, String jdbcPluginName) throws IllegalAccessException, InstantiationException, SQLException {
        try {
            DriverManager.getDriver(connectionString);
            return new DriverCleanup(null);
        }
        catch (SQLException e) {
            LOG.debug("Plugin Type: {} and Plugin Name: {}; Driver Class: {} not found. Registering JDBC driver via shim {} ", new Object[]{jdbcPluginType, jdbcPluginName, jdbcDriverClass.getName(), JDBCDriverShim.class.getName()});
            JDBCDriverShim driverShim = new JDBCDriverShim(jdbcDriverClass.newInstance());
            try {
                DBUtils.deregisterAllDrivers(jdbcDriverClass);
            }
            catch (ClassNotFoundException | NoSuchFieldException e1) {
                LOG.error("Unable to deregister JDBC Driver class {}", jdbcDriverClass);
            }
            DriverManager.registerDriver(driverShim);
            return new DriverCleanup(driverShim);
        }
    }

    public static List<Schema.Field> getSchemaFields(ResultSet resultSet, @Nullable String schemaStr) throws SQLException {
        Schema resultsetSchema = Schema.recordOf((String)"resultset", DBUtils.getSchemaFields(resultSet));
        if (!Strings.isNullOrEmpty((String)schemaStr)) {
            Schema schema;
            try {
                schema = Schema.parseJson((String)schemaStr);
            }
            catch (IOException e) {
                throw new IllegalArgumentException(String.format("Unable to parse schema string '%s'.", schemaStr), e);
            }
            for (Schema.Field field : schema.getFields()) {
                Schema simpleSchema;
                Schema.Field resultsetField = resultsetSchema.getField(field.getName());
                if (resultsetField == null) {
                    throw new IllegalArgumentException(String.format("Schema field '%s' is not present in input record.", field.getName()));
                }
                Schema resultsetFieldSchema = resultsetField.getSchema().isNullable() ? resultsetField.getSchema().getNonNullable() : resultsetField.getSchema();
                if (resultsetFieldSchema.equals((Object)(simpleSchema = field.getSchema().isNullable() ? field.getSchema().getNonNullable() : field.getSchema()))) continue;
                throw new IllegalArgumentException(String.format("Schema field '%s' has type '%s' but in input record found type '%s'.", field.getName(), simpleSchema.getDisplayName(), resultsetFieldSchema.getDisplayName()));
            }
            return schema.getFields();
        }
        return resultsetSchema.getFields();
    }

    public static List<Schema.Field> getSchemaFields(ResultSet resultSet) throws SQLException {
        ArrayList schemaFields = Lists.newArrayList();
        ResultSetMetaData metadata = resultSet.getMetaData();
        for (int i = 1; i <= metadata.getColumnCount(); ++i) {
            String columnName = metadata.getColumnName(i);
            int columnSqlType = metadata.getColumnType(i);
            int columnSqlPrecision = metadata.getPrecision(i);
            int columnSqlScale = metadata.getScale(i);
            String columnTypeName = metadata.getColumnTypeName(i);
            Schema columnSchema = DBUtils.getSchema(columnTypeName, columnSqlType, columnSqlPrecision, columnSqlScale);
            if (1 == metadata.isNullable(i)) {
                columnSchema = Schema.nullableOf((Schema)columnSchema);
            }
            Schema.Field field = Schema.Field.of((String)columnName, (Schema)columnSchema);
            schemaFields.add(field);
        }
        return schemaFields;
    }

    private static Schema getSchema(String typeName, int sqlType, int precision, int scale) throws SQLException {
        Schema.Type type = Schema.Type.STRING;
        switch (sqlType) {
            case 0: {
                type = Schema.Type.NULL;
                break;
            }
            case -8: {
                break;
            }
            case -7: 
            case 16: {
                type = Schema.Type.BOOLEAN;
                break;
            }
            case -6: 
            case 5: {
                type = Schema.Type.INT;
                break;
            }
            case 4: {
                type = "int unsigned".equalsIgnoreCase(typeName) ? Schema.Type.LONG : Schema.Type.INT;
                break;
            }
            case -5: {
                type = Schema.Type.LONG;
                break;
            }
            case 6: 
            case 7: {
                type = Schema.Type.FLOAT;
                break;
            }
            case 2: 
            case 3: {
                type = scale != 0 ? Schema.Type.DOUBLE : (precision > 9 ? Schema.Type.LONG : Schema.Type.INT);
                break;
            }
            case 8: {
                type = Schema.Type.DOUBLE;
                break;
            }
            case 91: {
                return Schema.of((Schema.LogicalType)Schema.LogicalType.DATE);
            }
            case 92: {
                return Schema.of((Schema.LogicalType)Schema.LogicalType.TIME_MICROS);
            }
            case 93: {
                return Schema.of((Schema.LogicalType)Schema.LogicalType.TIMESTAMP_MICROS);
            }
            case -4: 
            case -3: 
            case -2: 
            case 2004: {
                type = Schema.Type.BYTES;
                break;
            }
            case 70: 
            case 1111: 
            case 2000: 
            case 2001: 
            case 2002: 
            case 2003: 
            case 2006: 
            case 2009: {
                throw new SQLException((Throwable)new UnsupportedTypeException("Unsupported SQL Type: " + sqlType));
            }
        }
        return Schema.of((Schema.Type)type);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public static Object transformValue(int sqlType, int precision, int scale, ResultSet resultSet, String fieldName) throws SQLException {
        Object original = resultSet.getObject(fieldName);
        if (original != null) {
            switch (sqlType) {
                case -6: 
                case 5: {
                    return ((Number)original).intValue();
                }
                case 2: 
                case 3: {
                    BigDecimal decimal = (BigDecimal)original;
                    if (scale != 0) {
                        return decimal.doubleValue();
                    }
                    if (precision > 9) {
                        return decimal.longValue();
                    }
                    return decimal.intValue();
                }
                case 91: {
                    return resultSet.getDate(fieldName);
                }
                case 92: {
                    return resultSet.getTime(fieldName);
                }
                case 93: {
                    return resultSet.getTimestamp(fieldName);
                }
                case -8: {
                    return resultSet.getString(fieldName);
                }
                case 2004: {
                    Blob blob = (Blob)original;
                    try {
                        byte[] byArray = blob.getBytes(1L, (int)blob.length());
                        return byArray;
                    }
                    finally {
                        blob.free();
                    }
                }
                case 2005: {
                    Clob clob = (Clob)original;
                    try {
                        String string = clob.getSubString(1L, (int)clob.length());
                        return string;
                    }
                    finally {
                        clob.free();
                    }
                }
            }
        }
        return original;
    }

    public static void deregisterAllDrivers(Class<? extends Driver> driverClass) throws NoSuchFieldException, IllegalAccessException, ClassNotFoundException {
        Field field = DriverManager.class.getDeclaredField("registeredDrivers");
        field.setAccessible(true);
        List list = (List)field.get(null);
        for (Object driverInfo : list) {
            Class<?> driverInfoClass = DBUtils.class.getClassLoader().loadClass("java.sql.DriverInfo");
            Field driverField = driverInfoClass.getDeclaredField("driver");
            driverField.setAccessible(true);
            Driver d = (Driver)driverField.get(driverInfo);
            if (d == null) {
                LOG.debug("Found null driver object in drivers list. Ignoring.");
                continue;
            }
            LOG.debug("Removing non-null driver object from drivers list.");
            ClassLoader registeredDriverClassLoader = d.getClass().getClassLoader();
            if (registeredDriverClassLoader == null) {
                LOG.debug("Found null classloader for default driver {}. Ignoring since this may be using system classloader.", (Object)d.getClass().getName());
                continue;
            }
            if (!d.getClass().getClassLoader().equals(driverClass.getClassLoader())) continue;
            LOG.debug("Removing default driver {} from registeredDrivers", (Object)d.getClass().getName());
            list.remove(driverInfo);
        }
    }

    private static void shutDownMySQLAbandonedConnectionCleanupThread(ClassLoader classLoader) {
        try {
            Class<?> mysqlCleanupThreadClass;
            try {
                mysqlCleanupThreadClass = classLoader.loadClass("com.mysql.jdbc.AbandonedConnectionCleanupThread");
            }
            catch (ClassNotFoundException e) {
                LOG.trace("Failed to load MySQL abandoned connection cleanup thread class. Presuming DB App is not being run with MySQL and ignoring", (Throwable)e);
                return;
            }
            Method shutdownMethod = mysqlCleanupThreadClass.getMethod("shutdown", new Class[0]);
            shutdownMethod.invoke(null, new Object[0]);
            LOG.debug("Successfully shutdown MySQL connection cleanup thread.");
        }
        catch (Throwable e) {
            LOG.warn("Failed to shutdown MySQL connection cleanup thread. Ignoring.", e);
        }
    }

    private static void unregisterOracleMBean(ClassLoader classLoader) {
        ObjectName oracleJdbcMBeanName;
        try {
            classLoader.loadClass("oracle.jdbc.driver.OracleDriver");
        }
        catch (ClassNotFoundException e) {
            LOG.debug("Oracle JDBC Driver not found. Presuming that the DB App is not being run with an Oracle DB. Not attempting to cleanup Oracle MBean.");
            return;
        }
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        Hashtable<String, String> keys = new Hashtable<String, String>();
        keys.put("type", "diagnosability");
        keys.put("name", classLoader.getClass().getName() + "@" + Integer.toHexString(classLoader.hashCode()).toLowerCase());
        try {
            oracleJdbcMBeanName = new ObjectName("com.oracle.jdbc", keys);
        }
        catch (MalformedObjectNameException e) {
            LOG.debug("Exception while constructing Oracle JDBC MBean Name. Aborting cleanup.", (Throwable)e);
            return;
        }
        try {
            mbs.getMBeanInfo(oracleJdbcMBeanName);
        }
        catch (InstanceNotFoundException e) {
            LOG.debug("Oracle JDBC MBean not found. No cleanup necessary.");
            return;
        }
        catch (IntrospectionException | ReflectionException e) {
            LOG.debug("Exception while attempting to retrieve Oracle JDBC MBean. Aborting cleanup.", (Throwable)e);
            return;
        }
        try {
            mbs.unregisterMBean(oracleJdbcMBeanName);
            LOG.debug("Oracle MBean unregistered successfully.");
        }
        catch (InstanceNotFoundException | MBeanRegistrationException e) {
            LOG.debug("Exception while attempting to cleanup Oracle JDBCMBean. Aborting cleanup.", (Throwable)e);
        }
    }

    private DBUtils() {
        throw new AssertionError((Object)"Should not instantiate static utility class.");
    }
}

