/*
 * Decompiled with CFR 0.152.
 */
package io.contextmap.scanner.versioncontrol;

import io.contextmap.model.CommitDetail;
import io.contextmap.model.CommitOverview;
import io.contextmap.scanner.versioncontrol.VersionControlLogger;
import io.contextmap.scanner.versioncontrol.VersionControlScan;
import io.contextmap.scanner.versioncontrol.model.Tag;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.LogCommand;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectReader;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevObject;
import org.eclipse.jgit.revwalk.RevTag;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;

public class VersionControlScanner {
    private final VersionControlLogger logger;

    public VersionControlScanner(VersionControlLogger logger) {
        this.logger = logger;
    }

    public VersionControlScan scan() {
        VersionControlScan scan = new VersionControlScan();
        Git git = null;
        try {
            git = this.createGit();
        }
        catch (Exception e) {
            this.logger.warn("Unable to find git repository");
            return scan;
        }
        scan.commitDetailList = this.getCommits(git, true);
        scan.commitOverviewList = this.convertDetailsToOverviews(scan.commitDetailList);
        scan.tagList = this.getTags(git);
        return scan;
    }

    private List<Tag> getTags(Git git) {
        try {
            List<Ref> localTags;
            try {
                this.logger.debug("Getting local tags");
                localTags = git.tagList().call();
                for (Ref localTag : localTags) {
                    this.logger.debug("Found local branch " + localTag.getName());
                }
            }
            catch (Exception e) {
                this.logger.warn("Unable to fetch local tags");
                localTags = Collections.emptyList();
            }
            Repository repository = git.getRepository();
            ArrayList<Tag> tags = new ArrayList<Tag>();
            for (Ref ref : localTags) {
                Tag tag = new Tag();
                tag.originalRef = ref;
                tags.add(tag);
                Ref peeledRef = repository.getRefDatabase().peel(ref);
                tag.peeledRef = peeledRef.getPeeledObjectId() != null ? peeledRef : ref;
                tag.objectId = this.getRefObjectId(tag.peeledRef);
                this.enrichTagWithInfo(repository, tag);
            }
            Collections.sort(tags);
            if (!tags.isEmpty()) {
                Tag firstTag = (Tag)tags.get(0);
                Iterable logs = git.log().add((AnyObjectId)firstTag.objectId).call();
                firstTag.commits = this.getCommits(logs, false);
                this.logger.debug("Got " + firstTag.commits.size() + " commits for tag " + firstTag.name);
            }
            for (int i = 0; i < tags.size() - 1; ++i) {
                Tag previousTag = (Tag)tags.get(i);
                Tag currentTag = (Tag)tags.get(i + 1);
                LogCommand logCommand = git.log().addRange((AnyObjectId)previousTag.objectId, (AnyObjectId)currentTag.objectId);
                Iterable logs = logCommand.call();
                currentTag.previousTagName = previousTag.name;
                currentTag.commits = this.getCommits(logs, false);
                this.logger.debug("Got " + currentTag.commits.size() + " commits for tag " + currentTag.name);
            }
            this.logger.info("Added " + tags.size() + " tags");
            return tags;
        }
        catch (Exception e) {
            this.logger.warn("Unable to get tags");
            return Collections.emptyList();
        }
    }

    private void enrichTagWithInfo(Repository repository, Tag tag) throws Exception {
        tag.name = tag.originalRef.getName();
        try (ObjectReader reader = repository.newObjectReader();
             RevWalk revWalk = new RevWalk(reader);){
            ObjectId firstTagObjectId = tag.peeledRef.getObjectId();
            RevObject revObject = revWalk.parseAny((AnyObjectId)firstTagObjectId);
            if (revObject.getType() == 4) {
                RevTag revTag = revWalk.parseTag((AnyObjectId)firstTagObjectId);
                tag.createdOn = revTag.getTaggerIdent().getWhen();
                tag.name = revTag.getTagName();
            } else {
                this.logger.warn("Unable to walk tag since not recognized as tag (please check that all tags are fetched)");
            }
        }
    }

    private ObjectId getRefObjectId(Ref ref) {
        if (ref.getPeeledObjectId() != null) {
            return ref.getPeeledObjectId();
        }
        return ref.getObjectId();
    }

    private List<CommitOverview> convertDetailsToOverviews(List<CommitDetail> details) {
        HashMap<String, CommitOverview> dayToCounterMap = new HashMap<String, CommitOverview>();
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        for (CommitDetail commitDetail : details) {
            String key = formatter.format(commitDetail.day);
            CommitOverview commitOverview = (CommitOverview)dayToCounterMap.get(key);
            if (commitOverview == null) {
                commitOverview = new CommitOverview();
                commitOverview.day = commitDetail.day;
                dayToCounterMap.put(key, commitOverview);
            }
            ++commitOverview.amount;
        }
        return new ArrayList<CommitOverview>(dayToCounterMap.values());
    }

    private List<CommitDetail> getCommits(Git git, boolean limitToRecentCommits) {
        List<CommitDetail> commitDetails = new ArrayList<CommitDetail>();
        try {
            Iterable log = git.log().call();
            commitDetails = this.getCommits(log, limitToRecentCommits);
            this.logger.info("Added " + commitDetails.size() + " git commits");
        }
        catch (Exception e) {
            this.logger.warn("Unable to scan git commits");
        }
        return commitDetails;
    }

    private List<CommitDetail> getCommits(Iterable<RevCommit> log, boolean limitToRecentCommits) {
        List<Object> commitDetails = new ArrayList<CommitDetail>();
        int maxLengthCommitMessage = 150;
        int maxAgeCommitInDays = 90;
        int maxNrOfCommitsToTrack = 255;
        Calendar calendar = Calendar.getInstance();
        calendar.add(5, -1 * maxAgeCommitInDays);
        Date dateAfterWhichCommitsAreAllowed = calendar.getTime();
        for (RevCommit rc : log) {
            boolean addCommit = false;
            PersonIdent committer = rc.getCommitterIdent();
            if (limitToRecentCommits) {
                if (committer != null && committer.getWhen() != null && committer.getWhen().after(dateAfterWhichCommitsAreAllowed)) {
                    addCommit = true;
                }
            } else if (committer != null && committer.getWhen() != null) {
                addCommit = true;
            }
            if (!addCommit) continue;
            CommitDetail commitDetail = new CommitDetail();
            commitDetail.day = committer.getWhen();
            commitDetail.commitMessage = rc.getFullMessage();
            if (commitDetail.commitMessage != null && commitDetail.commitMessage.length() > maxLengthCommitMessage) {
                commitDetail.commitMessage = commitDetail.commitMessage.substring(0, maxLengthCommitMessage);
            }
            if (rc.getId().getName() != null) {
                commitDetail.shaHash = rc.getId().getName();
                if (commitDetail.shaHash.length() > 8) {
                    commitDetail.shaHash = commitDetail.shaHash.substring(0, 8);
                }
            }
            commitDetails.add(commitDetail);
        }
        if (commitDetails.size() > maxNrOfCommitsToTrack) {
            commitDetails = commitDetails.subList(0, maxNrOfCommitsToTrack);
        }
        return commitDetails;
    }

    private Git createGit() throws IOException {
        Repository repository = ((FileRepositoryBuilder)((FileRepositoryBuilder)new FileRepositoryBuilder().setMustExist(true)).findGitDir()).build();
        return new Git(repository);
    }
}

