package io.github.cdimascio.dotenv

import io.github.cdimascio.dotenv.internal.DotenvParser
import io.github.cdimascio.dotenv.internal.DotenvReader
import io.github.cdimascio.dotenv.internal.DotenvStreamReader
import java.io.File
import java.io.InputStream

abstract class Dotenv {
    companion object Instance {
        @JvmStatic fun configure(): DotenvBuilder = DotenvBuilder()
        @JvmStatic fun load(): Dotenv = DotenvBuilder().load()
    }

    operator abstract fun get(envVar: String): String?
}

class DotEnvException(message: String) : Exception(message)

class DotenvBuilder internal constructor() {
    private var filename = ".env"
    private var directoryPath = ""
    private var throwIfMissing = true
    private var throwIfMalformed = true
    private var inputStream: InputStream? = null

    @Deprecated("directory has been deprecated in favor of path")
    fun directory(path: String = directoryPath): DotenvBuilder {
        directoryPath = path
        return this
    }

    fun path(path: String = directoryPath): DotenvBuilder {
        val file = File(path)
        if (file.isDirectory) {
            directoryPath = file.path
        } else {
            directoryPath = file.parent
            filename = "${file.name}.${file.extension}"
        }
        return this
    }

    fun stream(stream: InputStream): DotenvBuilder {
        inputStream = stream
        return this
    }

    fun ignoreIfMissing(): DotenvBuilder {
        throwIfMissing = false
        return this
    }

    fun ignoreIfMalformed(): DotenvBuilder {
        throwIfMalformed = false
        return this
    }

    fun load(): Dotenv {
        if (inputStream != null && !directoryPath.isEmpty())
            throw IllegalArgumentException("stream and directory are mutually exclusive")

        val reader =
                if (inputStream != null) DotenvStreamReader(inputStream!!)
                else DotenvReader(directoryPath, filename)

        val parser = DotenvParser(
                reader,
                throwIfMalformed,
                throwIfMissing)
        val env = parser.parse()
        return DotenvImpl(env)
    }
}

private class DotenvImpl(envVars: List<Pair<String, String>>) : Dotenv() {
    private val map = envVars.associateBy({ it.first }, { it.second })

    override fun get(envVar: String): String? = System.getenv(envVar) ?: map[envVar]
}