package io.github.deemru.abcode;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;

/**
 * Universal alphabet converter with built-in Base58 support.
 *
 * @see <a href="https://github.com/deemru/ABCode-Java">Documentation</a>
 */
public class ABCode
{
    private final byte[] a;
    private final int aq;
    private final int[] amap;
    private final byte[] b;
    private final int bq;
    private final int[] bmap;

    private static final String BASE58_ALPHABET = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz";
    private static volatile ABCode base58Instance;

    /**
     * Creates ABCode instance with encoding alphabet and binary (256-byte) base.
     *
     * @param abc encoding alphabet (single-byte characters only)
     */
    public ABCode( String abc )
    {
        this( abc, null );
    }

    /**
     * Creates ABCode instance with encoding alphabet and custom base alphabet.
     *
     * @param abc  encoding alphabet (single-byte characters only)
     * @param base base alphabet, or null for binary (0-255)
     */
    public ABCode( String abc, String base )
    {
        this.a = abc.getBytes( StandardCharsets.ISO_8859_1 );
        this.aq = this.a.length;
        this.amap = map( this.a );

        if( base == null )
        {
            byte[] b = new byte[256];
            for( int i = 0; i < 256; i++ )
                b[i] = (byte)i;
            this.b = b;
            this.bq = 256;
            this.bmap = null;
        }
        else
        {
            this.b = base.getBytes( StandardCharsets.ISO_8859_1 );
            this.bq = this.b.length;
            this.bmap = map( this.b );
        }
    }

    /**
     * Returns a shared Base58 (Bitcoin) encoder/decoder instance.
     *
     * @return ABCode instance configured for Base58
     */
    public static ABCode base58()
    {
        if( base58Instance == null )
        {
            synchronized( ABCode.class )
            {
                if( base58Instance == null )
                    base58Instance = new ABCode( BASE58_ALPHABET );
            }
        }
        return base58Instance;
    }

    /**
     * Encodes binary data to the encoding alphabet.
     *
     * @param data binary data to encode
     * @return encoded string, or null on failure
     */
    public String encode( byte[] data )
    {
        byte[] result = abcode( data, b, bq, bmap, a, aq );
        return result == null ? null : new String( result, StandardCharsets.ISO_8859_1 );
    }

    /**
     * Encodes UTF-8 string to the encoding alphabet.
     *
     * @param data UTF-8 string to encode
     * @return encoded string, or null on failure
     */
    public String encode( String data )
    {
        return encode( data.getBytes( StandardCharsets.UTF_8 ) );
    }

    /**
     * Decodes string from the encoding alphabet to binary data.
     *
     * @param data encoded string to decode
     * @return decoded binary data, or null on failure
     */
    public byte[] decode( String data )
    {
        return abcode( data.getBytes( StandardCharsets.ISO_8859_1 ), a, aq, amap, b, bq );
    }

    /**
     * Decodes string from the encoding alphabet to UTF-8 string.
     *
     * @param data encoded string to decode
     * @return decoded UTF-8 string, or null on failure
     */
    public String decodeToString( String data )
    {
        byte[] result = decode( data );
        return result == null ? null : new String( result, StandardCharsets.UTF_8 );
    }

    private static int[] map( byte[] abc )
    {
        int[] map = new int[256];
        for( int i = 0; i < 256; i++ )
            map[i] = -1;
        for( int i = 0; i < abc.length; i++ )
            map[abc[i] & 0xFF] = i;
        return map;
    }

    private static byte[] abcode( byte[] data, byte[] from, int fromq, int[] frommap, byte[] to, int toq )
    {
        int n = data.length;
        int z = 0;
        for( ; z < n; z++ )
        {
            if( data[z] != from[0] )
                break;
        }

        if( z == n )
        {
            byte[] result = new byte[n];
            for( int i = 0; i < n; i++ )
                result[i] = to[0];
            return result;
        }

        if( fromq != 256 )
        {
            for( int i = z; i < n; i++ )
            {
                if( frommap[data[i] & 0xFF] == -1 )
                    return null;
            }
        }

        byte[] converted = convert( data, z, n, fromq, frommap, toq, to );

        if( z == 0 )
            return converted;

        byte[] result = new byte[z + converted.length];
        for( int i = 0; i < z; i++ )
            result[i] = to[0];
        System.arraycopy( converted, 0, result, z, converted.length );
        return result;
    }

    private static byte[] convert( byte[] data, int start, int n, int fromq, int[] frommap, int toq, byte[] to )
    {
        BigInteger bi;

        if( fromq == 256 )
            bi = new BigInteger( 1, data );
        else
        {
            long max = Long.MAX_VALUE / fromq - 1;
            long t = frommap[data[start] & 0xFF];
            long tq = fromq;
            bi = null;
            int i = start + 1;

            for( ; i < n; i++ )
            {
                t = t * fromq + frommap[data[i] & 0xFF];
                tq *= fromq;

                if( tq > max )
                {
                    bi = bi == null
                        ? BigInteger.valueOf( t )
                        : bi.multiply( BigInteger.valueOf( tq ) ).add( BigInteger.valueOf( t ) );

                    if( ++i == n )
                    {
                        tq = 1;
                        break;
                    }

                    t = frommap[data[i] & 0xFF];
                    tq = fromq;
                }
            }

            if( tq != 1 )
            {
                bi = bi == null
                    ? BigInteger.valueOf( t )
                    : bi.multiply( BigInteger.valueOf( tq ) ).add( BigInteger.valueOf( t ) );
            }
        }

        if( toq == 256 )
        {
            byte[] bytes = bi.toByteArray();
            if( bytes.length > 1 && bytes[0] == 0 )
            {
                byte[] result = new byte[bytes.length - 1];
                System.arraycopy( bytes, 1, result, 0, result.length );
                return result;
            }
            return bytes;
        }

        byte[] buf = new byte[bi.bitLength()];
        int len = 0;
        BigInteger base = BigInteger.valueOf( toq );

        while( bi.signum() > 0 )
        {
            BigInteger[] divRem = bi.divideAndRemainder( base );
            bi = divRem[0];
            buf[len++] = to[divRem[1].intValue()];
        }

        byte[] result = new byte[len];
        for( int i = 0; i < len; i++ )
            result[i] = buf[len - 1 - i];
        return result;
    }
}
