package graphql.execution;

import com.google.common.collect.ImmutableList;
import graphql.GraphQLError;
import graphql.Internal;
import graphql.PublicApi;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import static graphql.Assert.assertNotNull;

/**
 * The result object for {@link graphql.execution.DataFetcherExceptionHandler}s
 */
@PublicApi
public class DataFetcherExceptionHandlerResult<T>  {

    private final T data;
    private final List<GraphQLError> errors;
    private final Object localContext;

    /**
     * Creates a data fetcher result
     *
     * @param data   the data
     * @param errors the errors
     *
     * @deprecated use the {@link #newResult()} builder instead
     */
    @Internal
    @Deprecated
    public DataFetcherExceptionHandlerResult(T data, List<GraphQLError> errors) {
        this(data, errors, null);
    }

    private DataFetcherExceptionHandlerResult(T data, List<GraphQLError> errors, Object localContext) {
        this.data = data;
        this.errors = ImmutableList.copyOf(assertNotNull(errors));
        this.localContext = localContext;
    }

    /**
     * @return The data fetched. May be null.
     */
    public T getData() {
        return data;
    }

    /**
     * @return errors encountered when fetching data.  This will be non null but possibly empty.
     */
    public List<GraphQLError> getErrors() {
        return errors;
    }

    /**
     * @return true if there are any errors present
     */
    public boolean hasErrors() {
        return !errors.isEmpty();
    }

    /**
     * A data fetcher result can supply a context object for that field that is passed down to child fields
     *
     * @return a local context object
     */
    public Object getLocalContext() {
        return localContext;
    }

    /**
     * This helps you transform the current DataFetcherExceptionHandlerResult into another one by starting a builder with all
     * the current values and allows you to transform it how you want.
     *
     * @param builderConsumer the consumer code that will be given a builder to transform
     *
     * @return a new instance produced by calling {@code build} on that builder
     */

    /**
     * Creates a new data fetcher result builder
     *
     * @param <T> the type of the result
     *
     * @return a new builder
     */
    public static <T> DataFetcherExceptionHandlerResult.Builder<T> newResult() {
        return new DataFetcherExceptionHandlerResult.Builder<>();
    }

    public static class Builder<T> {
        private T data;
        private Object localContext;
        private final List<GraphQLError> errors = new ArrayList<>();

        public Builder(DataFetcherExceptionHandlerResult<T> existing) {
            data = existing.getData();
            localContext = existing.getLocalContext();
            errors.addAll(existing.getErrors());
        }

        public Builder(T data) {
            this.data = data;
        }

        public Builder() {
        }

        public DataFetcherExceptionHandlerResult.Builder<T> data(T data) {
            this.data = data;
            return this;
        }

        public DataFetcherExceptionHandlerResult.Builder<T> errors(List<GraphQLError> errors) {
            this.errors.addAll(errors);
            return this;
        }

        public DataFetcherExceptionHandlerResult.Builder<T> error(GraphQLError error) {
            this.errors.add(error);
            return this;
        }

        /**
         * @return true if there are any errors present
         */
        public boolean hasErrors() {
            return !errors.isEmpty();
        }

        public DataFetcherExceptionHandlerResult.Builder<T> localContext(Object localContext) {
            this.localContext = localContext;
            return this;
        }

        public DataFetcherExceptionHandlerResult<T> build() {
            return new DataFetcherExceptionHandlerResult<>(data, errors, localContext);
        }
    }
}
