package io.split.client;

import io.split.client.metrics.CachedMetrics;
import io.split.client.metrics.FireAndForgetMetrics;
import io.split.client.metrics.HttpMetrics;
import io.split.engine.experiments.SplitChangeFetcher;
import io.split.engine.experiments.SplitParser;
import io.split.engine.experiments.RefreshableSplitFetcherProvider;
import io.split.engine.impressions.CachedTreatmentLog;
import io.split.engine.metrics.Metrics;
import io.split.engine.segments.RefreshableSegmentFetcher;
import io.split.engine.segments.SegmentChangeFetcher;
import io.split.engine.segments.SegmentFetcher;
import io.codigo.grammar.Treatments;
import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.jackson.JacksonFeature;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.WebTarget;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Random;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Builds an instance of SplitClient.
 */
public class SplitClientBuilder {

    private static Random RANDOM = new Random();

    private static AtomicReference<SplitClient> _client = new AtomicReference<SplitClient>();
    private static Object _lock = new Object();

    public static SplitClient build(String apiToken) throws IOException {
        return build(apiToken, SplitClientConfig.builder().build());
    }


    public static SplitClient build(String apiToken, SplitClientConfig config) throws IOException {
        if (_client.get() != null) {
            return _client.get();
        }

        synchronized (_lock) {
            // double check locking.
            if (_client.get() != null) {
                return _client.get();
            }

            if (LocalhostSplitClientBuilder.LOCALHOST.equals(apiToken)) {
                SplitClient splitClient = LocalhostSplitClientBuilder.build();
                _client.set(splitClient);
                return splitClient;
            }

            Client client = ClientBuilder.newClient(new ClientConfig()
                    .property(ClientProperties.CONNECT_TIMEOUT, config.connectionTimeoutInMs())
                    .property(ClientProperties.READ_TIMEOUT, config.readTimeoutInMs())
                    .register(AddSplitHeadersFilter.instance(apiToken))
                    .register(ObjectMapperProvider.class)
                    .register(JacksonFeature.class));

            WebTarget rootTarget = client.target(config.endpoint());

            HttpMetrics httpMetrics = HttpMetrics.create(rootTarget);
            FireAndForgetMetrics uncachedFireAndForget = FireAndForgetMetrics.instance(httpMetrics, 2, 1000);

            SegmentChangeFetcher segmentChangeFetcher = HttpSegmentChangeFetcher.create(rootTarget, uncachedFireAndForget);
            SegmentFetcher segmentFetcher = new RefreshableSegmentFetcher(segmentChangeFetcher, findPollingPeriod(RANDOM, config.pollForSegmentChangesEveryNSeconds()));
            SplitParser splitParser = new SplitParser(segmentFetcher);

            SplitChangeFetcher splitChangeFetcher = HttpSplitChangeFetcher.create(rootTarget, uncachedFireAndForget);
            RefreshableSplitFetcherProvider experimentFetcherProvider = new RefreshableSplitFetcherProvider(splitChangeFetcher, splitParser, findPollingPeriod(RANDOM, config.pollForFeatureChangesEveryNSeconds()));

            CachedTreatmentLog treatmentLog = CachedTreatmentLog.instance(config.treatmentLogCacheExpireNSecondsAfterWrite(), 1000, CachedTreatmentLogRemovalListener.create(rootTarget));


            CachedMetrics cachedMetrics = new CachedMetrics(httpMetrics, 100);
            Metrics cachedFireAndForgetMetrics = FireAndForgetMetrics.instance(cachedMetrics, 2, 1000);

            SplitClient splitClient = new SplitClientImpl(experimentFetcherProvider.getFetcher(), treatmentLog, cachedFireAndForgetMetrics);

            _client.set(splitClient);
            return splitClient;
        }

    }

    private static int findPollingPeriod(Random rand, int max) {
        int min = max/2;
        return rand.nextInt((max - min) + 1) + min;
    }


    public static void main(String ... args) throws IOException {

        if (args.length !=2 ) {
            System.out.println("Usage: CodigoBuilder <api_token>");
            System.exit(1);
            return;
        }

        SplitClientConfig config = SplitClientConfig.builder()
                .endpoint("http://localhost:8080/sdk/api")
                .build();

        SplitClient client = SplitClientBuilder.build(args[0], config);

        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(System.in));

            for (String line = reader.readLine(); line != null; line = reader.readLine()) {
                if ("exit".equals(line)) {
                    System.exit(0);
                }
                String[] userIdAndSplit = line.split(" ");

                if (userIdAndSplit.length != 2) {
                    System.out.println("Could not understand command");
                    continue;
                }

                boolean isOn = client.isOn("User", userIdAndSplit[0], userIdAndSplit[1]);

                System.out.println(isOn ? Treatments.ON : Treatments.OFF);
            }

        } catch(IOException io){
            io.printStackTrace();
        }
    }
}
