package io.split.client;

import io.codigo.dtos.SplitChange;
import io.split.engine.experiments.SplitChangeFetcher;
import io.split.engine.metrics.Metrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Created by adilaijaz on 5/30/15.
 */
public final class HttpSplitChangeFetcher implements SplitChangeFetcher {
    private static final Logger _log = LoggerFactory.getLogger(HttpSplitChangeFetcher.class);

    private static final String PREFIX = "splitChangeFetcher";

    private final WebTarget _target;

    private static final String SINCE = "since";

    private final Metrics _metrics;

    public static HttpSplitChangeFetcher create(WebTarget root, Metrics metrics) {
        return new HttpSplitChangeFetcher(root.path("splitChanges"), metrics);
    }

    private HttpSplitChangeFetcher(WebTarget target, Metrics metrics) {
        _target = target;
        _metrics = metrics;
        checkNotNull(_target);
    }

    @Override
    public SplitChange fetch(long since) {

        long start = System.currentTimeMillis();

        try {
            Response response = _target
                    .queryParam(SINCE, since)
                    .request(MediaType.APPLICATION_JSON)
                    .acceptEncoding("gzip")
                    .get();

            if (response.getStatus() <  200 || response.getStatus() >= 300) {
                _log.error("Response status was: " + response.getStatus());
                _metrics.count(PREFIX + ".status." + response.getStatus(), 1);
                throw new RuntimeException("Could not retrieve split changes; http return code " + response.getStatus());
            }

            try {
                return response.readEntity(SplitChange.class);
            } catch (Throwable t) {
                // log error
                _log.error("Problem reading entity: " + response.getStatus(), t);
                _metrics.count(PREFIX + ".exception", 1);
                throw new RuntimeException(t);
            }

        } finally {
            _metrics.time(PREFIX + ".time", System.currentTimeMillis() - start);
        }
    }

}
