package io.split.client;

import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import io.split.client.dtos.KeyImpression;
import io.split.client.dtos.TestImpressions;
import io.split.client.utils.Utils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A cache removal listener that posts the removed value to a rest endpoint.
 *
 * @author adil
 */
public final class CachedTreatmentLogRemovalListener implements RemovalListener<String, List<KeyImpression>> {
    private static final Logger _log = LoggerFactory.getLogger(CachedTreatmentLogRemovalListener.class);

    private final CloseableHttpClient _client;
    private final URI _target;

    public static CachedTreatmentLogRemovalListener create(CloseableHttpClient client, URI root) throws URISyntaxException {
        return new CachedTreatmentLogRemovalListener(client, new URIBuilder(root).setPath("/api/testImpressions").build());
    }

    private CachedTreatmentLogRemovalListener(CloseableHttpClient client, URI target) {
        _client = client;
        _target = target;

        checkNotNull(_client);
        checkNotNull(_target);
    }


    @Override
    public void onRemoval(RemovalNotification<String, List<KeyImpression>> removalNotification) {
        String test = removalNotification.getKey();
        List<KeyImpression> keys = removalNotification.getValue();

        if (keys.isEmpty()) {
            _log.info("No impressions to report: " + test);
            return;
        }

        Set<String> keysSeen = Sets.newHashSet();
        List<KeyImpression> filtered = Lists.newArrayList();

        for (int i = (keys.size() - 1); i >= 0; i--) {
            KeyImpression keyImpression = keys.get(i);
            if (keysSeen.contains(keyImpression.keyName)) {
                continue;
            }

            keysSeen.add(keyImpression.keyName);
            filtered.add(keyImpression);
        }

        if (filtered.isEmpty()) {
            _log.info("No impressions to report post filtering: " + test);
            return;
        }

        if (_log.isDebugEnabled()) {
            _log.debug("Filteration: " + filtered.size() + " left from " + keys.size());
        }

        TestImpressions testImpressionsDTO = new TestImpressions();
        testImpressionsDTO.testName = test;
        testImpressionsDTO.keyImpressions = filtered;

        post(testImpressionsDTO);


    }

    private void post(TestImpressions dto) {

        CloseableHttpResponse response = null;

        try {
            StringEntity entity = Utils.toJsonEntity(dto);

            HttpPost request = new HttpPost(_target);
            request.setEntity(entity);

            response = _client.execute(request);

            int status = response.getStatusLine().getStatusCode();

            if (status < 200 || status >= 300) {
                _log.warn("Response status was: " + status);
            }

        } catch (Throwable t) {
            _log.warn("Exception when posting impressions" + dto, t);
        } finally {
            Utils.forceClose(response);
        }

    }


}
