/*
 * Decompiled with CFR 0.152.
 */
package io.streammachine.driver.client;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.streammachine.driver.client.AuthProvider;
import io.streammachine.driver.domain.Config;
import io.streammachine.driver.domain.StreamMachineException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.time.Instant;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.api.ContentProvider;
import org.eclipse.jetty.client.api.ContentResponse;
import org.eclipse.jetty.client.util.StringContentProvider;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AuthService {
    private static final Logger log = LoggerFactory.getLogger(AuthService.class);
    private static final ObjectMapper MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private final String billingId;
    private final String clientId;
    private final String clientSecret;
    private final HttpClient httpClient;
    private final CountDownLatch latch;
    private final Timer timer;
    private AuthProvider authProvider;
    private final String authUri;
    private final String refreshUri;

    public AuthService(String billingId, String clientId, String clientSecret, Config config) {
        this.billingId = billingId;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.httpClient = new HttpClient((SslContextFactory)new SslContextFactory.Client());
        try {
            this.httpClient.start();
        }
        catch (Exception e) {
            throw new IllegalStateException("An unexpected error occurred while starting a new AuthService for Stream Machine.", e);
        }
        try {
            this.authUri = new URI(String.format("%s://%s%s", config.getStsScheme(), config.getStsHost(), config.getStsAuthEndpoint())).toString();
            this.refreshUri = new URI(String.format("%s://%s%s", config.getStsScheme(), config.getStsHost(), config.getStsRefreshEndpoint())).toString();
        }
        catch (URISyntaxException e) {
            throw new IllegalStateException("Malformed URI(s) for " + this.getClass().getCanonicalName(), e);
        }
        this.timer = new Timer();
        this.latch = new CountDownLatch(1);
        this.timer.schedule((TimerTask)new AuthProviderInitializerTask(), 0L, Duration.ofMinutes(5L).toMillis());
        try {
            this.latch.await();
        }
        catch (InterruptedException e) {
            throw new IllegalStateException("Error while setting up authentication for Stream Machine", e);
        }
    }

    public String getAccessToken() {
        return this.authProvider.getIdToken();
    }

    public void stop() {
        try {
            this.timer.cancel();
            this.httpClient.stop();
        }
        catch (Exception e) {
            throw new StreamMachineException("Error stopping AuthService HttpClient", e);
        }
    }

    private void authenticate(String billingId, String clientId, String clientSecret) {
        try {
            ObjectNode payload = MAPPER.createObjectNode().put("billingId", billingId).put("clientId", clientId).put("clientSecret", clientSecret);
            this.doPost(this.authUri, payload);
        }
        catch (IOException | InterruptedException | ExecutionException | TimeoutException e) {
            log.error("An error occurred while requesting an access token with clientId '{}' and billingId '{}'", new Object[]{clientId, billingId, e});
        }
    }

    private void refresh(String refreshToken, String billingId, String clientId, String clientSecret) {
        try {
            ObjectNode payload = MAPPER.createObjectNode();
            payload.put("refreshToken", refreshToken);
            this.doPost(this.refreshUri, payload);
        }
        catch (IOException | InterruptedException | ExecutionException | TimeoutException e) {
            log.debug("Failed to refresh token with clientId '{}' and billingId '{}'", (Object)clientId, (Object)billingId);
            log.debug("Trying to request a new token with clientId '{}' and billingId '{}'", (Object)clientId, (Object)billingId);
            this.authenticate(billingId, clientId, clientSecret);
        }
    }

    private void doPost(String uri, ObjectNode payload) throws IOException, InterruptedException, TimeoutException, ExecutionException {
        ContentResponse response = this.httpClient.POST(uri).content((ContentProvider)new StringContentProvider(MAPPER.writeValueAsString((Object)payload))).header(HttpHeader.CONTENT_TYPE, "application/json; charset=UTF-8").send();
        this.authProvider = (AuthProvider)MAPPER.readValue(response.getContentAsString(), AuthProvider.class);
    }

    public static AuthServiceBuilder builder() {
        return new AuthServiceBuilder();
    }

    public static class AuthServiceBuilder {
        private String billingId;
        private String clientId;
        private String clientSecret;
        private Config config;

        AuthServiceBuilder() {
        }

        public AuthServiceBuilder billingId(String billingId) {
            this.billingId = billingId;
            return this;
        }

        public AuthServiceBuilder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public AuthServiceBuilder clientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        public AuthServiceBuilder config(Config config) {
            this.config = config;
            return this;
        }

        public AuthService build() {
            return new AuthService(this.billingId, this.clientId, this.clientSecret, this.config);
        }

        public String toString() {
            return "AuthService.AuthServiceBuilder(billingId=" + this.billingId + ", clientId=" + this.clientId + ", clientSecret=" + this.clientSecret + ", config=" + this.config + ")";
        }
    }

    private class AuthProviderInitializerTask
    extends TimerTask {
        private final long expirationSlackTimeSeconds = Duration.ofMinutes(10L).getSeconds();

        private AuthProviderInitializerTask() {
        }

        @Override
        public void run() {
            if (AuthService.this.authProvider == null) {
                log.debug("Initializing a new Auth Provider");
                AuthService.this.authenticate(AuthService.this.billingId, AuthService.this.clientId, AuthService.this.clientSecret);
                AuthService.this.latch.countDown();
            } else if (this.isAlmostExpired(AuthService.this.authProvider.getExpiresAt())) {
                log.debug("Refreshing an existing Auth Provider");
                AuthService.this.refresh(AuthService.this.authProvider.getRefreshToken(), AuthService.this.billingId, AuthService.this.clientId, AuthService.this.clientSecret);
            }
        }

        private boolean isAlmostExpired(long expirationTime) {
            long currentTime = Instant.now().getEpochSecond();
            return currentTime + this.expirationSlackTimeSeconds >= expirationTime;
        }
    }
}

