/**
 * Start the client communicating to a MockServer at the specified host and port
 * for example:
 *
 *   var client = mockServerClient("localhost", 1080);
 *
 * @param host the host for the MockServer to communicate with
 * @param port the port for the MockServer to communicate with
 */
var mockServerClient = function (host, port) {
    "use strict";

    var xmlhttp = new XMLHttpRequest();
    var mockServerUrl = "http://" + host + ":" + port;
    /**
     * The default headers added to to the mocked response when using mockSimpleResponse(...)
     */
    var defaultResponseHeaders = [
        {"name": "Content-Type", "values": ["application/json; charset=utf-8"]},
        {"name": "Cache-Control", "values": ["no-cache, no-store"]}
    ];
    var createResponseMatcher = function (path) {
        var headers = [];
        if (window.location.href.match(/testId\=.*/g)) {
            headers = [
                {
                    "name": "Referer",
                    "values": [".*" + window.location.href.match(/testId\=.*/g)[0] + ".*"]
                }
            ];
        }
        return {
            method: "",
            path: path,
            body: "",
            headers: headers,
            cookies: [],
            parameters: []
        }
    };
    var createExpectation = function (path, responseBody, statusCode) {
        return {
            httpRequest: createResponseMatcher(path),
            httpResponse: {
                statusCode: statusCode || 200,
                body: JSON.stringify(responseBody),
                cookies: [],
                headers: defaultResponseHeaders,
                delay: {
                    timeUnit: "MICROSECONDS",
                    value: 0
                }
            },
            times: {
                remainingTimes: 1,
                unlimited: false
            }
        };
    };
    /**
     * Setup an expectation in the MockServer by specifying an expectation object
     * for example:
     *
     *   mockServerClient("localhost", 1080).mockAnyResponse(
     *       {
     *           'httpRequest': {
     *               'path': '/somePath',
     *               'body': {
     *                   'type': "STRING",
     *                   'value': 'someBody'
     *               }
     *           },
     *           'httpResponse': {
     *               'statusCode': 200,
     *               'body': Base64.encode(JSON.stringify({ name: 'first_body' })),
     *               'delay': {
     *                   'timeUnit': 'MILLISECONDS',
     *                   'value': 250
     *               }
     *           },
     *           'times': {
     *               'remainingTimes': 1,
     *               'unlimited': false
     *           }
     *       }
     *   );
     *
     * @param expectation the expectation to setup on the MockServer
     */
    var mockAnyResponse = function (expectation) {
        xmlhttp.open("PUT", mockServerUrl + "/expectation", false);
        xmlhttp.setRequestHeader("Content-Type", "application/json; charset=utf-8");
        xmlhttp.send(JSON.stringify(expectation));
        return _this;
    };
    /**
     * Setup an expectation in the MockServer without having to specify the full expectation object
     * for example:
     *
     *   mockServerClient("localhost", 1080).mockSimpleResponse('/somePath', { name: 'value' }, 203);
     *
     * @param path the path to match requests against
     * @param responseBody the response body to return if a request matches
     * @param statusCode the response code to return if a request matches
     */
    var mockSimpleResponse = function (path, responseBody, statusCode) {
        return mockAnyResponse(createExpectation(path, responseBody, statusCode));
    };
    /**
     * Override the default headers that are used to specify the response headers in mockSimpleResponse(...)
     * (note: if you use mockAnyResponse(...) the default headers are not used)
     * for example:
     *
     *   mockServerClient("localhost", 1080).setDefaultHeaders([
     *       {"name": "Content-Type", "values": ["application/json; charset=utf-8"]},
     *       {"name": "Cache-Control", "values": ["no-cache, no-store"]}
     *   ])
     *
     * @param headers the path to match requests against
     */
    var setDefaultHeaders = function (headers) {
        defaultResponseHeaders = headers;
    };
    /**
     * Verify a request has been sent for example:
     *
     *   client.verify({
     *      'method': 'POST',
     *      'path': '/somePath'
     *   });
     *
     * @param request the http request that must be matched for this verification to pass
     * @param count   the number of times this request must be matched
     * @param exact   true if the count is matched as "equal to" or false if the count is matched as "greater than or equal to"
     */
    var verify = function (request, count, exact) {
        if (count === undefined) {
            count = 1;
        }
        xmlhttp.open("PUT", mockServerUrl + "/verify", false);
        xmlhttp.send(JSON.stringify({
            "httpRequest": request,
            "times": {
                "count": count,
                "exact": exact
            }
        }));
        if (xmlhttp.status !== 202) {
            console && console.error && console.error(xmlhttp.responseText);
            throw xmlhttp.responseText;
        }
        return _this;
    };
    /**
     * Verify a sequence of requests has been sent for example:
     *
     *   client.verifySequence(
     *       {
     *          'method': 'POST',
     *          'path': '/first_request'
     *       },
     *       {
     *          'method': 'POST',
     *          'path': '/second_request'
     *       },
     *       {
     *          'method': 'POST',
     *          'path': '/third_request'
     *       }
     *   );
     *
     * @param arguments the list of http requests that must be matched for this verification to pass
     */
    var verifySequence = function () {
        xmlhttp.open("PUT", mockServerUrl + "/verifySequence", false);
        var requestSequence = [];
        for (var i = 0; i < arguments.length; i++) {
            requestSequence.push(arguments[i]);
        }
        xmlhttp.send(JSON.stringify({
            "httpRequests": requestSequence
        }));
        if (xmlhttp.status !== 202) {
            console && console.error && console.error(xmlhttp.responseText);
            throw xmlhttp.responseText;
        }
        return _this;
    };
    /**
     * Reset MockServer by clearing all expectations
     */
    var reset = function () {
        xmlhttp.open("PUT", mockServerUrl + "/reset", false);
        xmlhttp.send("");
        return _this;
    };
    /**
     * Clear all expectations that match the specified path
     *
     * @param path the path to decide which expectations to cleared
     */
    var clear = function (path) {
        xmlhttp.open("PUT", mockServerUrl + "/clear", false);
        xmlhttp.send(JSON.stringify(createResponseMatcher(path || ".*")));
        return _this;
    };
    /**
     * Pretty-print the json for all expectations for the specified path.
     * This is particularly helpful when debugging expectations. The expectation
     * are printed into a dedicated log called mockserver_request.log
     *
     * @param path the path to decide which expectations to dump to the log
     */
    var dumpToLogs = function (path) {
        xmlhttp.open("PUT", mockServerUrl + "/dumpToLog", false);
        xmlhttp.send(JSON.stringify(createResponseMatcher(path || ".*")));
        return _this;
    };

    var _this = {
        mockAnyResponse: mockAnyResponse,
        mockSimpleResponse: mockSimpleResponse,
        setDefaultHeaders: setDefaultHeaders,
        verify: verify,
        verifySequence: verifySequence,
        reset: reset,
        clear: clear,
        dumpToLogs: dumpToLogs
    };
    return  _this;
};