/*
 * Decompiled with CFR 0.152.
 */
package org.tinystruct.mail;

import jakarta.mail.Address;
import jakarta.mail.Message;
import jakarta.mail.MessagingException;
import jakarta.mail.Session;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.Temporal;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.tinystruct.ApplicationException;
import org.tinystruct.mail.Connection;
import org.tinystruct.mail.POP3Connection;
import org.tinystruct.mail.SMTPConnection;
import org.tinystruct.system.Configuration;

public final class ConnectionManager
implements AutoCloseable {
    private static final Logger logger = Logger.getLogger(ConnectionManager.class.getName());
    private final ConcurrentLinkedQueue<PooledConnection> idleConnections = new ConcurrentLinkedQueue();
    private final ConcurrentHashMap<Connection, Instant> activeConnections = new ConcurrentHashMap();
    private final ScheduledExecutorService cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread t = new Thread(r, "ConnectionManager-Cleanup");
        t.setDaemon(true);
        return t;
    });
    private final int maxPoolSize = Integer.parseInt(System.getProperty("mail.pool.size", "10"));
    private final Duration maxIdleTime = Duration.ofMinutes(Integer.parseInt(System.getProperty("mail.pool.idle.minutes", "5")));
    private final Duration maxLifetime = Duration.ofMinutes(Integer.parseInt(System.getProperty("mail.pool.lifetime.minutes", "30")));
    private volatile boolean isShutdown = false;

    private ConnectionManager() {
        this.cleanupExecutor.scheduleAtFixedRate(this::cleanup, 1L, 1L, TimeUnit.MINUTES);
    }

    public static ConnectionManager getInstance() {
        return SingletonHolder.INSTANCE;
    }

    public Connection getConnection(Configuration<String> config, Connection.PROTOCOL protocol) throws ApplicationException {
        if (this.isShutdown) {
            throw new ApplicationException("Connection manager is shut down");
        }
        PooledConnection connection = this.idleConnections.poll();
        if (connection != null) {
            if (this.isConnectionValid(connection)) {
                this.markConnectionActive(connection);
                return connection;
            }
            this.closeConnection(connection);
        }
        if (this.activeConnections.size() < this.maxPoolSize) {
            connection = this.createNewConnection(config, protocol);
            this.markConnectionActive(connection);
            return connection;
        }
        try {
            return this.waitForConnection(config, protocol);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ApplicationException("Interrupted while waiting for connection", e);
        }
    }

    public void releaseConnection(Connection connection) {
        if (connection instanceof PooledConnection) {
            PooledConnection pooledConnection = (PooledConnection)connection;
            this.activeConnections.remove(connection);
            if (this.isConnectionValid(pooledConnection)) {
                this.idleConnections.offer(pooledConnection);
            } else {
                this.closeConnection(pooledConnection);
            }
        }
    }

    @Override
    public void close() {
        this.isShutdown = true;
        this.cleanupExecutor.shutdown();
        try {
            if (!this.cleanupExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.cleanupExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.cleanupExecutor.shutdownNow();
        }
        this.idleConnections.forEach(this::closeConnection);
        this.idleConnections.clear();
        ((ConcurrentHashMap.KeySetView)this.activeConnections.keySet()).forEach(this::closeConnection);
        this.activeConnections.clear();
    }

    private void cleanup() {
        Instant now = Instant.now();
        this.idleConnections.removeIf(conn -> {
            if (this.isConnectionExpired((PooledConnection)conn, now)) {
                this.closeConnection((Connection)conn);
                return true;
            }
            return false;
        });
        this.activeConnections.entrySet().removeIf(entry -> {
            if (Duration.between((Temporal)entry.getValue(), now).compareTo(this.maxLifetime) > 0) {
                this.closeConnection((Connection)entry.getKey());
                return true;
            }
            return false;
        });
    }

    private boolean isConnectionValid(PooledConnection connection) {
        return connection != null && connection.getCreationTime() != null && connection.getLastUsedTime() != null && Duration.between(connection.getCreationTime(), Instant.now()).compareTo(this.maxLifetime) <= 0 && Duration.between(connection.getLastUsedTime(), Instant.now()).compareTo(this.maxIdleTime) <= 0;
    }

    private boolean isConnectionExpired(PooledConnection connection, Instant now) {
        return Duration.between(connection.getLastUsedTime(), now).compareTo(this.maxIdleTime) > 0 || Duration.between(connection.getCreationTime(), now).compareTo(this.maxLifetime) > 0;
    }

    private void closeConnection(Connection connection) {
        try {
            connection.close();
        }
        catch (MessagingException e) {
            logger.log(Level.WARNING, "Error closing connection", e);
        }
    }

    private void markConnectionActive(PooledConnection connection) {
        connection.setLastUsedTime(Instant.now());
        this.activeConnections.put(connection, Instant.now());
    }

    private PooledConnection createNewConnection(Configuration<String> config, Connection.PROTOCOL protocol) throws ApplicationException {
        Connection baseConnection = protocol == Connection.PROTOCOL.SMTP ? new SMTPConnection(config) : new POP3Connection(config);
        return new PooledConnection(baseConnection);
    }

    private Connection waitForConnection(Configuration<String> config, Connection.PROTOCOL protocol) throws InterruptedException, ApplicationException {
        for (int attempts = 0; attempts < 3; ++attempts) {
            PooledConnection connection = this.idleConnections.poll();
            if (connection != null && this.isConnectionValid(connection)) {
                this.markConnectionActive(connection);
                return connection;
            }
            Thread.sleep(1000L);
        }
        throw new ApplicationException("Connection pool exhausted");
    }

    private static final class SingletonHolder {
        static final ConnectionManager INSTANCE = new ConnectionManager();

        private SingletonHolder() {
        }
    }

    private static class PooledConnection
    implements Connection {
        private final Connection delegate;
        private final Instant creationTime;
        private volatile Instant lastUsedTime;

        PooledConnection(Connection delegate) {
            this.delegate = delegate;
            this.creationTime = Instant.now();
            this.lastUsedTime = Instant.now();
        }

        @Override
        public String getId() {
            return this.delegate.getId();
        }

        @Override
        public Connection.PROTOCOL getProtocol() {
            return this.delegate.getProtocol();
        }

        @Override
        public Session getSession() {
            return this.delegate.getSession();
        }

        @Override
        public boolean available() {
            return this.delegate.available();
        }

        @Override
        public void send(Message message, Address[] recipients) throws MessagingException {
            try {
                this.delegate.send(message, recipients);
                this.lastUsedTime = Instant.now();
            }
            catch (MessagingException e) {
                logger.log(Level.WARNING, "Failed to send message using connection: " + this.getId(), e);
                throw e;
            }
        }

        @Override
        public void close() throws MessagingException {
            try {
                this.delegate.close();
            }
            catch (MessagingException e) {
                logger.log(Level.WARNING, "Error closing connection: " + this.getId(), e);
                throw e;
            }
        }

        Instant getCreationTime() {
            return this.creationTime;
        }

        Instant getLastUsedTime() {
            return this.lastUsedTime;
        }

        void setLastUsedTime(Instant time) {
            this.lastUsedTime = time;
        }
    }
}

