/*
 * Decompiled with CFR 0.152.
 */
package ai.tecton.client.transport;

import ai.tecton.client.TectonClientOptions;
import ai.tecton.client.exceptions.TectonClientException;
import ai.tecton.client.exceptions.TectonServiceException;
import ai.tecton.client.transport.HttpRequest;
import ai.tecton.client.transport.HttpResponse;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.ConnectionPool;
import okhttp3.Dispatcher;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

public class TectonHttpClient {
    private HttpUrl url;
    private final String apiKey;
    private final OkHttpClient client;
    private final AtomicBoolean isClosed;
    private static final String API_KEY_PREFIX = "Tecton-key ";
    private static final String USER_AGENT_STRING_PREFIX = "tecton-http-java-client ";
    private static final Map<String, String> defaultHeaders = new HashMap<String, String>(){
        {
            this.put(HttpHeader.CONTENT_TYPE.getName(), MediaType.APPLICATION_JSON.getName());
            this.put(HttpHeader.ACCEPT.getName(), MediaType.APPLICATION_JSON.getName());
            this.put(HttpHeader.USER_AGENT.getName(), "tecton-http-java-client 0.9.1");
        }
    };

    public TectonHttpClient(String url, String apiKey, TectonClientOptions tectonClientOptions) {
        this.validateUrl(url);
        this.validateApiKey(apiKey);
        this.apiKey = apiKey;
        Dispatcher dispatcher = new Dispatcher();
        dispatcher.setMaxRequestsPerHost(tectonClientOptions.getMaxParallelRequests());
        OkHttpClient.Builder builder = new OkHttpClient.Builder().readTimeout(tectonClientOptions.getReadTimeout().getSeconds(), TimeUnit.SECONDS).connectTimeout(tectonClientOptions.getConnectTimeout().getSeconds(), TimeUnit.SECONDS).dispatcher(dispatcher);
        ConnectionPool connectionPool = new ConnectionPool(tectonClientOptions.getMaxIdleConnections(), tectonClientOptions.getKeepAliveDuration().getSeconds(), TimeUnit.SECONDS);
        builder.connectionPool(connectionPool);
        this.client = builder.build();
        this.isClosed = new AtomicBoolean(false);
    }

    public TectonHttpClient(String url, String apiKey, OkHttpClient httpClient) {
        this.validateUrl(url);
        if (apiKey != null) {
            this.validateApiKey(apiKey);
        }
        this.client = httpClient;
        this.apiKey = apiKey;
        this.isClosed = new AtomicBoolean(false);
    }

    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            this.client.dispatcher().executorService().shutdown();
            this.client.connectionPool().evictAll();
        }
    }

    public boolean isClosed() {
        return this.isClosed.get();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public HttpResponse performRequest(String endpoint, HttpMethod method, String requestBody) {
        HttpRequest httpRequest = new HttpRequest(this.url.url().toString(), endpoint, method, this.apiKey, requestBody);
        Request request = this.buildRequestWithDefaultHeaders(httpRequest);
        Call call = this.client.newCall(request);
        try (Response response = call.execute();){
            HttpResponse httpResponse = new HttpResponse(response);
            return httpResponse;
        }
        catch (Exception e) {
            throw new TectonClientException(e.getMessage());
        }
    }

    public List<HttpResponse> performParallelRequests(String endpoint, HttpMethod method, List<String> requestBodyList, Duration timeout) throws TectonClientException {
        final ParallelCallHandler parallelCallHandler = new ParallelCallHandler(requestBodyList.size());
        final List requestList = requestBodyList.parallelStream().map(requestBody -> new HttpRequest(this.url.url().toString(), endpoint, method, this.apiKey, (String)requestBody)).map(this::buildRequestWithDefaultHeaders).collect(Collectors.toList());
        final CountDownLatch countDownLatch = new CountDownLatch(requestBodyList.size());
        Callback callback = new Callback(){

            public void onFailure(Call call, IOException e) {
                if (!(e instanceof InterruptedIOException)) {
                    parallelCallHandler.logCallFailure(e.getMessage());
                }
            }

            public void onResponse(Call call, Response response) {
                try (ResponseBody responseBody = response.body();){
                    parallelCallHandler.set(requestList.indexOf(call.request()), new HttpResponse(response, responseBody));
                }
                catch (Exception e) {
                    throw new TectonServiceException(e.getMessage());
                }
                finally {
                    Objects.requireNonNull(response.body()).close();
                    countDownLatch.countDown();
                }
            }
        };
        requestList.parallelStream().forEach(request -> this.client.newCall(request).enqueue(callback));
        try {
            boolean completedAllCalls = countDownLatch.await(timeout.toMillis(), TimeUnit.MILLISECONDS);
            if (!parallelCallHandler.failureMessageList.isEmpty()) {
                throw new TectonClientException(String.format("Unable to perform call. %s", parallelCallHandler.failureMessageList.get(0)));
            }
            return parallelCallHandler.responseList;
        }
        catch (InterruptedException e) {
            throw new TectonClientException(e.getMessage());
        }
    }

    public Request buildRequestWithDefaultHeaders(HttpRequest httpRequest) {
        Request.Builder requestBuilder = new Request.Builder().url(httpRequest.getUrl());
        for (Map.Entry<String, String> header : defaultHeaders.entrySet()) {
            requestBuilder.header(header.getKey(), header.getValue());
        }
        String apiKeyHeader = StringUtils.join((Object[])new String[]{API_KEY_PREFIX + httpRequest.getApiKey()});
        requestBuilder.header(HttpHeader.AUTHORIZATION.getName(), apiKeyHeader);
        switch (httpRequest.getMethod()) {
            default: 
        }
        okhttp3.MediaType mediaType = okhttp3.MediaType.parse((String)MediaType.APPLICATION_JSON.getName());
        RequestBody requestBody = RequestBody.create((String)httpRequest.getJsonBody(), (okhttp3.MediaType)mediaType);
        requestBuilder.post(requestBody);
        return requestBuilder.build();
    }

    Duration getReadTimeout() {
        return Duration.ofMillis(this.client.readTimeoutMillis());
    }

    Duration getConnectTimeout() {
        return Duration.ofMillis(this.client.connectTimeoutMillis());
    }

    int getMaxParallelRequests() {
        return this.client.dispatcher().getMaxRequestsPerHost();
    }

    private void validateApiKey(String apiKey) {
        try {
            Validate.notEmpty((CharSequence)apiKey);
        }
        catch (Exception e) {
            throw new TectonClientException("API Key cannot be empty");
        }
    }

    private void validateUrl(String url) {
        try {
            Validate.notEmpty((CharSequence)url);
            this.url = HttpUrl.parse((String)url);
        }
        catch (Exception e) {
            throw new TectonClientException("Cannot connect to Tecton because the URL is invalid");
        }
    }

    static class ParallelCallHandler {
        List<HttpResponse> responseList;
        List<String> failureMessageList;

        ParallelCallHandler(int numberOfCalls) {
            this.responseList = new ArrayList<Object>(Collections.nCopies(numberOfCalls, null));
            this.failureMessageList = new ArrayList<String>(numberOfCalls);
        }

        void set(int index, HttpResponse httpResponse) {
            this.responseList.set(index, httpResponse);
        }

        void logCallFailure(String failureMessage) {
            this.failureMessageList.add(failureMessage);
        }
    }

    static enum MediaType {
        APPLICATION_JSON("application/json"),
        PLAIN_TEXT("text/plain");

        private final String name;

        private MediaType(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }
    }

    static enum HttpHeader {
        CONTENT_TYPE("Content-Type"),
        ACCEPT("Accept"),
        AUTHORIZATION("Authorization"),
        USER_AGENT("User-Agent");

        private final String name;

        private HttpHeader(String headerName) {
            this.name = headerName;
        }

        public String getName() {
            return this.name;
        }
    }

    public static enum HttpMethod {
        GET,
        POST,
        PUT,
        DELETE;

    }
}

