package ch.rhj.image.ico;

import java.awt.image.RenderedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteOrder;
import java.util.ArrayList;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.spi.ImageWriterSpi;
import javax.imageio.stream.ImageOutputStream;

public class IcoImageWriter extends ImageWriter {
	
	private final static int HEADER_SIZE = 6;
	private final static int ENTRY_SIZE = 16;
	
	private final ArrayList<IIOImage> images = new ArrayList<>();
	private final ArrayList<byte[]> buffers = new ArrayList<>();
	
	public IcoImageWriter(ImageWriterSpi originatingProvider) {
		
		super(originatingProvider);
	}

	@Override
	public IIOMetadata getDefaultStreamMetadata(ImageWriteParam param) {

		return null;
	}

	@Override
	public IIOMetadata getDefaultImageMetadata(ImageTypeSpecifier imageType, ImageWriteParam param) {

		return new IcoMetadata();
	}

	@Override
	public IIOMetadata convertStreamMetadata(IIOMetadata inData, ImageWriteParam param) {

		return null;
	}

	@Override
	public IIOMetadata convertImageMetadata(IIOMetadata inData, ImageTypeSpecifier imageType, ImageWriteParam param) {

		return null;
	}

	@Override
	public boolean canWriteRasters() {

		return false;
	}

	@Override
	public boolean canWriteSequence() {

		return true;
	}

	@Override
	public void prepareWriteSequence(IIOMetadata streamMetadata) throws IOException {

		images.clear();
		buffers.clear();
	}

	@Override
	public void endWriteSequence() throws IOException {

		ImageOutputStream stream = (ImageOutputStream) getOutput();
		
		stream.setByteOrder(ByteOrder.LITTLE_ENDIAN);
		
		try {
			
			createBuffers();
			writeHeader(stream);
			writeIndex(stream);
			writeImages(stream);
			stream.flush();
			
		} finally {
			
			buffers.clear();
		}
	}

	@Override
	public void writeToSequence(IIOImage image, ImageWriteParam param) throws IOException {

		images.add(image);
	}

	@Override
	public void write(IIOMetadata streamMetadata, IIOImage image, ImageWriteParam param) throws IOException {

		prepareWriteSequence(streamMetadata);
		writeToSequence(image, param);
		endWriteSequence();
	}
	
	private void createBuffers() throws IOException {
		
		for (IIOImage image : images) {
			
			ByteArrayOutputStream output = new ByteArrayOutputStream();
			
			ImageIO.write(image.getRenderedImage(), "png", output);
			buffers.add(output.toByteArray());
		}
	}
	
	private void writeHeader(ImageOutputStream stream) throws IOException {
		
		stream.writeShort(0); // reserved
		stream.writeShort(1); // ICO
		stream.writeShort(images.size());
	}
	
	private void writeIndex(ImageOutputStream stream) throws IOException {
		
		int offset = HEADER_SIZE + images.size() * ENTRY_SIZE;
		
		for (int i = 0, n = images.size(); i < n; ++i) {
			
			IIOImage image = images.get(i);
			RenderedImage rendered = image.getRenderedImage();
			int width = rendered.getWidth();
			int height = rendered.getHeight();
			int size = buffers.get(i).length;
			
			if (width > 255) width = 0;
			if (height > 255) height = 0;
			
			stream.write(width);
			stream.write(height);
			stream.write(0); // no color palette
			stream.write(0); // reserved
			stream.writeShort(1); // color planes
			stream.writeShort(32); // bits per pixel
			stream.writeInt(size);
			stream.writeInt(offset);
			
			offset += size;
		}
	}
	
	private void writeImages(ImageOutputStream stream) throws IOException {
		
		for (byte[] buffer : buffers)
			stream.write(buffer);
	}
}
