package ch.rhj.image.ico;

import static ch.rhj.image.ico.IcoConstants.VERSION;

import javax.imageio.metadata.IIOInvalidTreeException;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.metadata.IIOMetadataFormatImpl;
import javax.imageio.metadata.IIOMetadataNode;

import org.w3c.dom.Node;

public class IcoMetadata extends IIOMetadata implements Cloneable {
	
	public static final String NATIVE_METADATA_FORMAT_NAME = "ch_rhj_image_ico_" + VERSION;
	
	public int width;
	public int height;
	public short bitsPerPixel;
	
	public byte[] palette = null;
	public int paletteSize;
	
	public IcoMetadata() {
		
		super(
			true, // standardMetadataFormatSupported
			NATIVE_METADATA_FORMAT_NAME,
			IcoMetadataFormat.class.getName(),
			null, // extraMetadataFormatNames
			null // extraMetadataFormatClassNames
		);
	}

	@Override
	public boolean isReadOnly() {

		return true;
	}

	@Override
	public Node getAsTree(String formatName) {

		if (NATIVE_METADATA_FORMAT_NAME.equals(formatName))
			return getNativeTree();
		
		if (IIOMetadataFormatImpl.standardMetadataFormatName.equals(formatName))
			return getStandardTree();
		
		throw new IllegalArgumentException("metadata format '" + formatName + "' not supported");
	}
	
	@Override
	public void setFromTree(String formatName, Node root) throws IIOInvalidTreeException {

		throw new IllegalStateException("metadata is read-only");
	}

	@Override
	public void mergeTree(String formatName, Node root) throws IIOInvalidTreeException {

		throw new IllegalStateException("metadata is read-only");
	}
	
	@Override
	public void reset() {

		throw new IllegalStateException("metadata is read-only");
	}
	
	@Override
	protected IIOMetadataNode getStandardChromaNode() {

		IIOMetadataNode result = new IIOMetadataNode("Chroma");
		
		IIOMetadataNode colorSpaceTypeNode = new IIOMetadataNode("ColorSpaceType");
		
		colorSpaceTypeNode.setAttribute("name", "RGB");
		result.appendChild(colorSpaceTypeNode);
		
		IIOMetadataNode numChannelsNode = new IIOMetadataNode("NumChannels");
		
		numChannelsNode.setAttribute("value", "4");
		result.appendChild(numChannelsNode);
		
		IIOMetadataNode paletteNode = new IIOMetadataNode("Palette");
		
		for (int i = 0, j = 0; i < paletteSize; ++i) {
			
			IIOMetadataNode entryNode = new IIOMetadataNode("PaletteEntry");
			
			int blue = palette[j++] & 0xff;
			int green = palette[j++] & 0xff;
			int red = palette[j++] & 0xff;
			
			entryNode.setAttribute("index", Integer.toString(i));
			entryNode.setAttribute("blue", Integer.toString(blue));
			entryNode.setAttribute("green", Integer.toString(green));
			entryNode.setAttribute("red", Integer.toString(red));
			
			paletteNode.appendChild(entryNode);
		}
		
		result.appendChild(paletteNode);
		
		return result;
	}

	@Override
	protected IIOMetadataNode getStandardDataNode() {

		throw new RuntimeException("not yet implemented");
	}

	@Override
	protected IIOMetadataNode getStandardDimensionNode() {

		throw new RuntimeException("not yet implemented");
	}

	@Override
	protected IIOMetadataNode getStandardTransparencyNode() {

		throw new RuntimeException("not yet implemented");
	}

	@Override
	public IcoMetadata clone() {

		try {
			
			return (IcoMetadata) super.clone();
			
		} catch (CloneNotSupportedException e) {

			return null;
		}
	}

	private Node getNativeTree() {
		
		IIOMetadataNode root = new IIOMetadataNode(NATIVE_METADATA_FORMAT_NAME);
		
		IIOMetadataNode widthNode = new IIOMetadataNode("Width");
		IIOMetadataNode heightNode = new IIOMetadataNode("Height");
		IIOMetadataNode bitsPerPixelNode = new IIOMetadataNode("BitsPerPixel");
		
		widthNode.setUserObject(width);
		widthNode.setNodeValue(Integer.toString(width));
		
		heightNode.setUserObject(height);
		heightNode.setNodeValue(Integer.toString(height));
		
		bitsPerPixelNode.setUserObject(bitsPerPixel);
		bitsPerPixelNode.setNodeValue(Short.toString(bitsPerPixel));
		
		root.appendChild(widthNode);
		root.appendChild(heightNode);
		root.appendChild(bitsPerPixelNode);
		
		if ((palette != null) && (paletteSize > 0)) {
			
			IIOMetadataNode paletteNode = new IIOMetadataNode("Palette");
			
			root.appendChild(paletteNode);
			
			for (int i = 0, j = 0; i < paletteSize; i++) {
				
				IIOMetadataNode entryNode = new IIOMetadataNode("PaletteEntry");
				
				int blue = palette[j++] & 0xff;
				int green = palette[j++] & 0xff;
				int red = palette[j++] & 0xff;

				IIOMetadataNode blueNode = new IIOMetadataNode("Blue");
				IIOMetadataNode greenNode = new IIOMetadataNode("Green");
				IIOMetadataNode redNode = new IIOMetadataNode("Red");
				
				blueNode.setUserObject(blue);
				blueNode.setNodeValue(Integer.toString(blue));
				
				greenNode.setUserObject(green);
				greenNode.setNodeValue(Integer.toString(green));
				
				redNode.setUserObject(red);
				redNode.setNodeValue(Integer.toString(red));
				
				entryNode.appendChild(blueNode);
				entryNode.appendChild(greenNode);
				entryNode.appendChild(redNode);
				
				paletteNode.appendChild(entryNode);
			}
		}
		
		return root;
	}
}
