package co.datadome.api.common;

import org.apache.http.conn.DnsResolver;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

public class DataDomeResolver implements DnsResolver {

    private final String host;
    private final String proxyServerName;
    private final int refreshIn;

    private final DnsResolver dnsResolver;

    private Map<String, InetAddress[]> inetAddresses;

    private AtomicLong lastTimeRefreshed = new AtomicLong(Long.MIN_VALUE);

    public DataDomeResolver(String host, int refreshIn, DnsResolver dnsResolver) throws UnknownHostException {
        this(host, null, refreshIn, dnsResolver);
    }

    public DataDomeResolver(String host, String proxyServerName, int refreshIn, DnsResolver dnsResolver) throws UnknownHostException {
        this.host = host;
        this.proxyServerName = proxyServerName;
        this.refreshIn = refreshIn;
        this.dnsResolver = dnsResolver;
        this.inetAddresses = new HashMap<>(2);
        refresh();
    }

    public void refresh() throws UnknownHostException {
        for (String h : new String[]{this.host, this.proxyServerName}) {
            if (h != null && !h.trim().isEmpty()) {
                InetAddress[] resolved = dnsResolver.resolve(h);
                inetAddresses.put(h, resolved);
            }
        }
    }

    @Override
    public InetAddress[] resolve(String host) throws UnknownHostException {
        if (host != null && !(host.equals(this.host) || host.equals(this.proxyServerName))) {
            throw new UnknownHostException("DataDome resolver configured to resolve only " + this.host + "," + this.proxyServerName);
        }
        long lastTime = lastTimeRefreshed.get();
        long now = System.currentTimeMillis();
        if (refreshIn <= 0 || (now - refreshIn > lastTime && lastTimeRefreshed.compareAndSet(lastTime, now))) {
            refresh();
        }
        InetAddress[] result = inetAddresses.get(host);
        if (result == null) {
            throw new UnknownHostException("DataDome resolver failed to resolve this host: " + host);
        }
        return result;
    }
}
