package co.datadome.api.servlet;

import co.datadome.api.common.DataDomeResponse;
import co.datadome.api.common.HttpRequest;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import static co.datadome.api.common.DataDomeHeaders.TIME_SPENT;

public class FilterHttpRequest implements HttpRequest {

    private final EnrichedHttpServletRequest httpServletRequest;
    private final HttpServletResponse httpServletResponse;
    private final FilterChain filterChain;

    public FilterHttpRequest(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, FilterChain filterChain) {
        this.httpServletRequest = new EnrichedHttpServletRequest(httpServletRequest);
        this.httpServletResponse = httpServletResponse;
        this.filterChain = filterChain;
    }

    @Override
    public String uri() {
        return httpServletRequest.getRequestURI();
    }

    @Override
    public String query() {
        return httpServletRequest.getQueryString();
    }

    @Override
    public String protocol() {
        return httpServletRequest.getScheme();
    }

    @Override
    public String method() {
        return httpServletRequest.getMethod();
    }

    @Override
    public void next() throws IOException, ServletException {
        filterChain.doFilter(httpServletRequest, httpServletResponse);
    }

    @Override
    public void timeSpent(long elapsedTime) {
        httpServletRequest.setAttribute(TIME_SPENT, elapsedTime);
    }

    @Override
    public void addHeadersInRequest(Set<Map.Entry<String, String>> headers) {
        for (Map.Entry<String, String> header : headers) {
            httpServletRequest.addHeader(header.getKey(), header.getValue());
            httpServletRequest.setAttribute(header.getKey(), header.getValue());
        }
    }

    @Override
    public void addHeadersInResponse(Set<Map.Entry<String, String>> headers) {
        for (Map.Entry<String, String> header : headers) {
            httpServletResponse.addHeader(header.getKey(), header.getValue());
            httpServletRequest.setAttribute(header.getKey(), header.getValue());
        }
    }

    @Override
    public void block(DataDomeResponse dataDomeResponse) throws IOException {
        httpServletResponse.setStatus(dataDomeResponse.getStatusCode());
        httpServletResponse.getWriter().write(dataDomeResponse.getResponseBody()); // showing captcha
    }

    @Override
    public String getHeader(String header) {
        return httpServletRequest.getHeader(header);
    }

    @Override
    public Collection<String> headers() {
        return Collections.list(httpServletRequest.getHeaderNames());
    }

    @Override
    public String getIp() {
        return httpServletRequest.getRemoteAddr();
    }

    @Override
    public String getPort() {
        return Integer.toString(httpServletRequest.getRemotePort());
    }

    @Override
    public String getCookie(String cookieName) {
        Cookie[] cookies = httpServletRequest.getCookies();

        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals(cookieName)) {
                    return cookie.getValue();
                }
            }
        }

        return null;
    }
}
