package co.datadome.api.servlet;

import co.datadome.api.common.DataDomeEnvironment;
import co.datadome.api.common.DataDomeRequestConsumer;
import co.datadome.api.common.DataDomeService;

import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import java.io.IOException;
import java.net.UnknownHostException;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class DataDomeFilter implements Filter {

    private static final Logger logger = Logger.getLogger(DataDomeFilter.class.getCanonicalName());

    private DataDomeRequestConsumer requestConsumer = null;

    public void init(FilterConfig filterConfig) throws ServletException {
        String apiKey = getStringParam(filterConfig, "datadome.apikey", null);
        String apiHost = getStringParam(filterConfig, "datadome.hostname", DataDomeService.DEFAULT_API_HOST);
        boolean apiSSL = getBooleanParam(filterConfig, "datadome.ssl", DataDomeService.DEFAULT_API_SSL);
        String regex = getStringParam(filterConfig, "datadome.regex", DataDomeService.DEFAULT_REGEX);
        String exclusionRegex = getStringParam(filterConfig, "datadome.exclusion_regex", DataDomeService.DEFAULT_EXCLUSION_REGEX);

        int connectTimeout = getIntegerParam(filterConfig, "datadome.connection_timeout", DataDomeService.DEFAULT_CONNECT_TIMEOUT);
        int readTimeout = getIntegerParam(filterConfig, "datadome.read_timeout", DataDomeService.DEFAULT_READ_TIMEOUT);
        int maxConnections = getIntegerParam(filterConfig, "datadome.max_connections", DataDomeService.DEFAULT_MAX_TOTAL_CONNECTIONS);

        String proxyServer = getStringParam(filterConfig, "datadome.proxy_server", "");
        int proxyPort = getIntegerParam(filterConfig, "datadome.proxy_port", 0);
        boolean proxySSL = getBooleanParam(filterConfig, "datadome.proxy_ssl", false);

        String skipIps = getStringParam(filterConfig, "datadome.skip_ips", "");
        boolean useXForwardedHost = getBooleanParam(filterConfig, "datadome.use_x_forwarded_host", false);
        boolean useForwarded = getBooleanParam(filterConfig, "datadome.use_forwarded", false);
        Collection<String> skipIpSet = skipIps.isEmpty() ? Collections.<String>emptyList() : Arrays.asList(skipIps.split(","));

        try {
            requestConsumer = new DataDomeRequestConsumer(
                    new DataDomeService(apiKey, apiHost, apiSSL, proxyServer, proxyPort, proxySSL, connectTimeout, readTimeout, maxConnections),
                    regex, exclusionRegex, skipIpSet, useXForwardedHost, useForwarded);
        } catch (UnknownHostException e) {
            throw new ServletException("Can not create DataDomeService", e);
        }

        DataDomeEnvironment.setServerInfo(filterConfig.getServletContext().getServerInfo());
    }

    protected void init(DataDomeRequestConsumer dataDomeRequestConsumer, String serverInfo) {
        this.requestConsumer = dataDomeRequestConsumer;
        DataDomeEnvironment.setServerInfo(serverInfo);
    }

    public void doFilter(final ServletRequest req, final ServletResponse resp, final FilterChain chain)
            throws IOException, ServletException {

        if (!(req instanceof HttpServletRequest) || !(resp instanceof HttpServletResponse)) {
            logger.log(Level.WARNING, "DataDome can process only HttpServletRequest");
            chain.doFilter(req, resp);
            return;
        }

        FilterHttpRequest httpRequest = new FilterHttpRequest((HttpServletRequest) req, (HttpServletResponse) resp, chain);
        requestConsumer.accept(httpRequest);
    }

    public void destroy() {
        requestConsumer = null;
    }

    private static final Pattern varRegex = Pattern.compile("(\\$\\{(/?[^\\}]+)\\})");

    protected String getStringParam(FilterConfig filterConfig, String name, String defaultValue) throws UnavailableException {
        String rawValue = filterConfig.getInitParameter(name);
        if (rawValue == null) {
            if (defaultValue == null) {
                throw new UnavailableException("Missed DataDome filter init-param: " + name);
            }
            return defaultValue;
        }

        Matcher m = varRegex.matcher(rawValue);

        while (m.find()) {
            String wrappedEnvName = m.group(1);
            String envName = m.group(2);
            String envValue = getEnv(envName);
            if (envValue == null) {
                throw new UnavailableException("Undefined variable " + envName);
            } else {
                rawValue = rawValue.replace(wrappedEnvName, envValue);
            }
        }

        return rawValue;
    }

    protected String getEnv(String envName) {
        return System.getenv(envName);
    }

    protected boolean getBooleanParam(FilterConfig filterConfig, String name, Boolean defaultValue) throws UnavailableException {
        return Boolean.valueOf(getStringParam(filterConfig, name, defaultValue.toString()));
    }

    protected int getIntegerParam(FilterConfig filterConfig, String name, Integer defaultValue) throws UnavailableException {
        String param = getStringParam(filterConfig, name, defaultValue.toString());
        try {
            return Integer.valueOf(param);
        } catch (NumberFormatException e) {
            throw new UnavailableException("DataDome filter init-param: " + name + " should be valid integer but it was: " + param);
        }
    }

    protected DataDomeRequestConsumer getRequestConsumer() {
        return requestConsumer;
    }

}
