package co.datadome.api.common;

import jakarta.servlet.ServletException;
import java.io.IOException;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import static co.datadome.api.common.DataDomeHeaders.*;

public class DataDomeRequestConsumer {

    private static final Logger logger = Logger.getLogger(DataDomeRequestConsumer.class.getSimpleName());

    private final DataDomeService dataDomeService;

    public Pattern getRegex() {
        return regex;
    }

    public Pattern getExclusionRegex() {
        return exclusionRegex;
    }

    private final Pattern regex;
    private final Pattern exclusionRegex;
    private final Pattern exclusionHostRegex;

    private final List<IpAddressMatcher> subnetMatchers;
    private final boolean useXForwardedHost;
    private final boolean useForwarded;

    public boolean isUseXForwardedHost() {
        return useXForwardedHost;
    }

    public DataDomeRequestConsumer(DataDomeService dataDomeService, String regex, String exclusionRegex, Collection<String> skipIps,
                                   boolean useXForwardedHost) {
        this(dataDomeService, regex, exclusionRegex, null, skipIps, useXForwardedHost, false);
    }

    public DataDomeRequestConsumer(DataDomeService dataDomeService, String regex, String exclusionRegex, String exclusionHostRegEx, Collection<String> skipIps,
                                   boolean useXForwardedHost) {
        this(dataDomeService, regex, exclusionRegex, exclusionHostRegEx, skipIps, useXForwardedHost, false);
    }

    public DataDomeRequestConsumer(DataDomeService dataDomeService, String regex, String exclusionRegex, String exclusionHostRegex, Collection<String> skipIps,
                                   boolean useXForwardedHost, boolean useForwarded) {
        this.dataDomeService = dataDomeService;
        this.subnetMatchers = convertToSubnet(skipIps);

        this.regex = nullOrEmpty(regex) ? null : Pattern.compile(regex);
        this.exclusionRegex = nullOrEmpty(exclusionRegex) ? null : Pattern.compile(exclusionRegex);
        this.exclusionHostRegex = nullOrEmpty(exclusionHostRegex) ? null : Pattern.compile(exclusionHostRegex);
        this.useXForwardedHost = useXForwardedHost;
        this.useForwarded = useForwarded;
    }

    protected static List<IpAddressMatcher> convertToSubnet(Collection<String> subnets) {
        List<IpAddressMatcher> addressMatchers = new ArrayList<>();
        for (String subnet : subnets) {
            addressMatchers.add(new IpAddressMatcher(subnet));
        }
        return addressMatchers;
    }

    protected static boolean matchSubnets(String ip, List<IpAddressMatcher> subnetMatchers) {
        for (IpAddressMatcher matcher : subnetMatchers) {
            if (matcher.matches(ip)) {
                return true;
            }
        }
        return false;
    }

    private static boolean nullOrEmpty(String value) {
        return value == null || value.length() == 0;
    }

    public void accept(final HttpRequest httpRequest) throws IOException, ServletException {
        DataDomeRequest dataDomeRequest = buildDataDomeRequest(httpRequest);

        if (!isRegexMatched(dataDomeRequest.getUri())) {
            logger.log(Level.FINE, "DataDome regex miss");
            httpRequest.next();
            return;
        }

        if (matchSubnets(httpRequest.getIp(), subnetMatchers)) {
            logger.log(Level.FINE, "DataDome skip IP: {0}", httpRequest.getIp().replaceAll("[\n\r\t]", "_"));
            httpRequest.next();
            return;
        }

        if (matchHostExclusionRegExp(httpRequest.getHeader(HOST_HEADER))) {
            logger.log(Level.FINE, "DataDome skip Host: {0}", httpRequest.getHeader(HOST_HEADER).replaceAll("[\n\r\t]", "_"));
            httpRequest.next();
            return;
        }

        DataDomeResponse dataDomeResponse = validateRequest(dataDomeRequest, httpRequest);

        if (dataDomeResponse == null) {
            httpRequest.next();
            return;
        }

        httpRequest.addHeadersInRequest(dataDomeResponse.getRequestHeaders().entrySet());
        httpRequest.addHeadersInResponse(dataDomeResponse.getResponseHeaders().entrySet());

        // block if 403 and show captcha
        if (dataDomeResponse.shouldBeBlocked()) {
            httpRequest.block(dataDomeResponse);
            return;
        }

        httpRequest.next();
    }

    protected boolean matchHostExclusionRegExp(String hostName) {
        if (hostName == null) {
            return false;
        }
        return exclusionHostRegex != null && exclusionHostRegex.matcher(hostName).find();
    }


    protected boolean isRegexMatched(String uri) {
        if (uri == null) {
            return false;
        }

        if (exclusionRegex != null && exclusionRegex.matcher(uri).find()) {
            return false;
        }

        if (regex != null) {
            return regex.matcher(uri).find();
        }

        return true;
    }

    private DataDomeResponse validateRequest(DataDomeRequest dataDomeRequest, HttpRequest httpRequest) throws IOException {
        long startTime = System.currentTimeMillis();

        DataDomeResponse dataDomeResponse = dataDomeService.validateRequest(dataDomeRequest);
        long elapsedTime = System.currentTimeMillis() - startTime;
        logger.log(Level.FINE, "DataDome request/response time in milliseconds: {0}", elapsedTime);

        httpRequest.timeSpent(elapsedTime);

        return dataDomeResponse;
    }

    public DataDomeRequest buildDataDomeRequest(HttpRequest request) {
        DataDomeRequest.Builder requestBuilder = getRequestBuilder();

        requestBuilder.setUserAgent(request.getHeader(USER_AGENT_HEADER));
        requestBuilder.setIp(request.getIp());
        requestBuilder.setPort(request.getPort());

        String xfProto = request.getHeader(X_FORWARDED_PROTO);
        if (xfProto != null && (xfProto.equalsIgnoreCase("http") || xfProto.equalsIgnoreCase("https"))) {
            requestBuilder.setProtocol(xfProto);
        } else {
            requestBuilder.setProtocol(request.protocol());
        }
        requestBuilder.setForwardedForIP(request.getHeader(X_FORWARDED_FOR_HEADER));

        requestBuilder.setHost(request.getHeader(HOST_HEADER));

        if (this.useForwarded) {
            // attempt to read from "forwarded" header
            Map<String, String> forwardedHeaderMap = parseForwardedHeader(request.getHeader(FORWARDED));
            if(forwardedHeaderMap.containsKey("host")) {
                requestBuilder.setHost(forwardedHeaderMap.get("host"));
            }
            if(forwardedHeaderMap.containsKey("for")) {
                requestBuilder.setForwardedForIP(forwardedHeaderMap.get("for"));
            }
            if(forwardedHeaderMap.containsKey("proto")) {
                requestBuilder.setProtocol(forwardedHeaderMap.get("proto"));
            }
        } else if (this.useXForwardedHost) {
            String forwardedHost = request.getHeader(X_FORWARDED_HOST_HEADER);
            if (forwardedHost != null) {
                requestBuilder.setHost(forwardedHost);
            }
        }
        requestBuilder.setReferer(request.getHeader(REFERER_HEADER));

        String clientID = request.getHeader(X_DATADOME_CLIENTID);
        boolean shouldUseXSetCookie = clientID != null && !clientID.isEmpty();

        requestBuilder.setClientID(shouldUseXSetCookie ? clientID : request.getCookie(DATADOME_COOKIE));
        requestBuilder.setxSetCookie(shouldUseXSetCookie);

        requestBuilder.setUri(request.uri());
        requestBuilder.setRequest(uriQuery(request.uri(), request.query()));


        requestBuilder.setMethod(request.method());
        requestBuilder.setCookiesLen(Integer.toString(getHeaderLen(request, COOKIE_HEADER)));

        // Java 9 has `Instant.now()` with up to nanoseconds resolution but here we should support an old one
        requestBuilder.setTimeRequest(Long.toString(getRequestTimeStampInMicro()));

        requestBuilder.setServerHostname(request.getHeader(HOST_HEADER));
        requestBuilder.setPostParamLen(request.getHeader(CONTENT_LENGTH_HEADER));

        requestBuilder.setHeadersList(headerList(request.headers()));

        requestBuilder.setAuthorizationLen(Integer.toString(getHeaderLen(request, AUTHORIZATION_HEADER)));
        requestBuilder.setxRequestedWith(request.getHeader(X_REQUESTED_WITH_HEADER));
        requestBuilder.setOrigin(request.getHeader(ORIGIN_HEADER));
        requestBuilder.setConnection(request.getHeader(CONNECTION_HEADER));
        requestBuilder.setPragma(request.getHeader(PRAGMA_HEADER));
        requestBuilder.setCacheControl(request.getHeader(CACHE_CONTROL_HEADER));
        requestBuilder.setContentType(request.getHeader(CONTENT_TYPE_HEADER));
        requestBuilder.setFrom(request.getHeader(FROM_HEADER));
        requestBuilder.setxRealIP(request.getHeader(X_REAL_IP_HEADER));
        requestBuilder.setVia(request.getHeader(VIA_HEADER));
        requestBuilder.setTrueClientIP(request.getHeader(TRUE_CLIENT_IP_HEADER));
        requestBuilder.setAccept(request.getHeader(ACCEPT_HEADER));
        requestBuilder.setAcceptCharset(request.getHeader(ACCEPT_CHARSET_HEADER));
        requestBuilder.setAcceptEncoding(request.getHeader(ACCEPT_ENCODING_HEADER));
        requestBuilder.setAcceptLanguage(request.getHeader(ACCEPT_LANGUAGE_HEADER));
        requestBuilder.setSecCHUA(request.getHeader(SEC_CH_UA_HEADER));
        requestBuilder.setSecCHUAArch(request.getHeader(SEC_CH_UA_ARCH_HEADER));
        requestBuilder.setSecCHUAFullVersionList(request.getHeader(SEC_CH_UA_FULL_VERSION_LIST_HEADER));
        requestBuilder.setSecCHUAPlatform(request.getHeader(SEC_CH_UA_PLATFORM_HEADER));
        requestBuilder.setSecCHUAModel(request.getHeader(SEC_CH_UA_MODEL_HEADER));
        requestBuilder.setSecCHUAMobile(request.getHeader(SEC_CH_UA_MOBILE_HEADER));
        requestBuilder.setSecCHDeviceMemory(request.getHeader(SEC_CH_DEVICE_MEMORY_HEADER));
        requestBuilder.setSecFetchDest(request.getHeader(SEC_FETCH_DEST));
        requestBuilder.setSecFetchMode(request.getHeader(SEC_FETCH_MODE));
        requestBuilder.setSecFetchSite(request.getHeader(SEC_FETCH_SITE));
        requestBuilder.setSecFetchUser(request.getHeader(SEC_FETCH_USER));

        return requestBuilder.build();
    }

    private static int getHeaderLen(HttpRequest request, String name) {
        String value = request.getHeader(name);

        return value == null ? 0 : value.length();
    }


    public DataDomeRequest.Builder getRequestBuilder() {
        return DataDomeRequest.builder();
    }

    public long getRequestTimeStampInMicro() {
        return System.currentTimeMillis() * 1000;
    }

    public List<IpAddressMatcher> getSubnetMatchers() {
        return subnetMatchers;
    }

    /**
     * Parse the forwarded header and returns a map with its fields
     * @param forwarded - header value
     * @return Map
     */
    protected static Map<String, String> parseForwardedHeader(String forwarded) {
        Map<String, String> result = new HashMap<>();
        if (forwarded != null) {
            String[] forwardedElements = forwarded.toLowerCase().split(";");
            for (String element : forwardedElements) {
                String[] multiList = element.split(",");
                for (String uniqueElement : multiList) {
                    String[] items = uniqueElement.split("=");
                    if (items.length > 1) {
                        String key = items[0].trim();
                        String newValue = items[1];
                        result.merge(key, newValue, (oldValue,v) -> oldValue + "," + v);
                    }
                }
            }
        }
        return result;
    }
}
