package com.abtasty.flagship.api;

import com.abtasty.flagship.utils.FlagshipConstants;
import com.abtasty.flagship.utils.FlagshipLogManager;
import com.abtasty.flagship.utils.LogManager;

import java.net.MalformedURLException;
import java.net.URL;

/**
 * Interface to implement in order to completely transform url
 */
interface IFlagshipServiceEndpoints {
    String overrideCampaignUrl(String url);
}

/**
 * This class define all the Flagship services endpoints that may be called by the SDK, and allows their customization.
 * <p>
 * ----------------------------------------------------------------------------------------------------------------------
 * <br><br><b>- Decision service</b> is dedicated to the campaign allocation, campaign activation (when a flag is exposed)
 * and segment (context) pushing.
 * <br><br>Calls:<br><br>
 * <b>https://decision.flagship.io/v2/%s/campaigns?exposeAllKeys=true</b>
 * <br> => Called at each fetchFlags in DecisionAPI config mode
 * for campaign allocation.<br>
 * <b>https://decision.flagship.io/v2/activate</b>
 * <br> => Called when a flag is exposed to user in any config modes.
 * <b>https://decision.flagship.io/v2/%s/events</b>
 * <br> => Called at each fetchFlags in Bucketing config mode for pushing context segments.
 * segments.
 * <br><br>
 * ----------------------------------------------------------------------------------------------------------------------
 * <br><br><b>- Bucketing service</b> is dedicated for bucketing decision file downloading.
 * <br><br>Calls:<br><br>
 * <b>https://cdn.flagship.io/%s/bucketing.json</b>
 * <br> => Called at each polling intervals in bucketing config mode.
 * <br><br>
 * ----------------------------------------------------------------------------------------------------------------------
 * <br><br><b>- Tracking service</b> is dedicated for hit tracking and data reporting.
 * <br><br>Calls:<br><br>
 * <b>https://events.flagship.io</b>
 * <br> => Called whenever a Hit is sent.
 * <br><br>
 */
public class FlagshipServiceEndpoints implements IFlagshipServiceEndpoints {

    public static final String DEFAULT_DECISION_DOMAIN       = "https://decision.flagship.io";
    public static final String DEFAULT_BUCKETING_DOMAIN      = "https://cdn.flagship.io"; //%s/bucketing.json
    public static final String DEFAULT_TRACKING_DOMAIN       = "https://events.flagship.io";

    public static final String DEFAULT_CAMPAIGNS_ENDPOINT    = "/v2/%s/campaigns?exposeAllKeys=true";
    public static final String DEFAULT_ACTIVATE_ENDPOINT     = "/v2/activate";
    public static final String DEFAULT_EVENTS_ENDPOINT       = "/v2/%s/events"; //For bucketing only.
    public static final String DEFAULT_BUCKETING_ENDPOINT    = "/%s/bucketing.json";
    public static final String CONTEXT_PARAM                 = "&sendContextEvent=false";

//    public enum FLAGSHIP_DOMAINS {
//        DECISION("https://decision.flagship.io"),
//        BUCKETING("https://cdn.flagship.io"),
//        TRACKING("https://events.flagship.io"),
//        ;
//
//        private final String url;
//
//        FLAGSHIP_DOMAINS(String url) {
//            this.url = url;
//        }
//
//        public String getUrl() {
//            return url;
//        }
//    }
//
//    public enum FLAGSHIP_ENDPOINTS {
//        CAMPAIGNS("/v2/%s/campaigns?exposeAllKeys=true"),
//        ACTIVATE("/v2/activate"),
//        EVENTS("/v2/%s/events"),
//        BUCKETING("/%s/bucketing.json")
//        ;
//
//        private final String url;
//
//        FLAGSHIP_ENDPOINTS(String url) {
//            this.url = url;
//        }
//
//        public String getUrl() {
//            return url;
//        }
//    }

//    protected String customCampaignDomain       = DEFAULT_DECISION_DOMAIN;
//    protected String customActivateDomain       = DEFAULT_DECISION_DOMAIN;
//    protected String customEventDomain          = DEFAULT_DECISION_DOMAIN;
//    protected String customBucketingDomain      = DEFAULT_BUCKETING_DOMAIN;
//    protected String customTrackingDomain       = DEFAULT_TRACKING_DOMAIN;
//
//    protected String customCampaignEndpoint     = DEFAULT_CAMPAIGNS_ENDPOINT;
//    protected String customActivateEndpoint     = DEFAULT_ACTIVATE_ENDPOINT;
//    protected String customEventEndpoint        = DEFAULT_EVENTS_ENDPOINT;
//    protected String customBucketingEndpoint    = DEFAULT_BUCKETING_ENDPOINT;
//    protected String customTrackingEndpoint     = "";

    protected String customCampaignEndpoint = DEFAULT_DECISION_DOMAIN + DEFAULT_CAMPAIGNS_ENDPOINT;
    protected String customActivateEndpoint = DEFAULT_DECISION_DOMAIN + DEFAULT_ACTIVATE_ENDPOINT;
    protected String customEventEndpoint = DEFAULT_DECISION_DOMAIN + DEFAULT_EVENTS_ENDPOINT;
    protected String customBucketingEndpoint = DEFAULT_BUCKETING_DOMAIN + DEFAULT_BUCKETING_ENDPOINT;
    protected String customTrackingEndpoint = DEFAULT_TRACKING_DOMAIN;


    protected String envId            = "_ENV_ID_";

    public final void init(String envId) {
        if (envId != null && !envId.isEmpty())
            this.envId = envId;
        if (customCampaignEndpoint.equals(DEFAULT_DECISION_DOMAIN + DEFAULT_CAMPAIGNS_ENDPOINT))
            customCampaignEndpoint = String.format(DEFAULT_CAMPAIGNS_ENDPOINT, envId);
        if (customEventEndpoint.equals(DEFAULT_DECISION_DOMAIN + DEFAULT_EVENTS_ENDPOINT))
            customEventEndpoint = String.format(DEFAULT_EVENTS_ENDPOINT, envId);
        if (customBucketingEndpoint.equals(DEFAULT_BUCKETING_DOMAIN + DEFAULT_BUCKETING_ENDPOINT))
            customBucketingEndpoint = String.format(DEFAULT_BUCKETING_ENDPOINT, envId);
//        if (customCampaignEndpoint.equals(DEFAULT_DECISION_DOMAIN + DEFAULT_CAMPAIGNS_ROUTE))
//            customCampaignEndpoint = String.format(customCampaignEndpoint, envId);
//        if (customEventEndpoint.equals(DEFAULT_DECISION_DOMAIN + DEFAULT_EVENTS_ROUTE))
//            customEventEndpoint = String.format(customEventEndpoint, envId);
//        if (customBucketingEndpoint.equals(DEFAULT_BUCKETING_DOMAIN + DEFAULT_BUCKETING_ROUTE))
//            customBucketingEndpoint = String.format(customBucketingEndpoint, envId);
    }

    public final String getCampaignUrl(Boolean consent) {
//        String customUrl = (customCampaignDomain + customCampaignEndpoint);
        String customUrl = (customCampaignEndpoint);
        if (!isUrlValid(customUrl)) {
            String fallbackUrl = String.format(DEFAULT_DECISION_DOMAIN + DEFAULT_CAMPAIGNS_ENDPOINT, envId)
                    + (!consent ? CONTEXT_PARAM : "");
            logUrlInvalidError(customUrl, fallbackUrl);
            return fallbackUrl;
        }
        return customUrl;
//        if (isUrlValid(customCampaignEndpoint))
//            return customCampaignEndpoint + (!consent ? CONTEXT_PARAM : "");
//        else {
//            String fallbackUrl = String.format(DEFAULT_DECISION_DOMAIN + DEFAULT_CAMPAIGNS_ENDPOINT, envId)
//                    + (!consent ? CONTEXT_PARAM : "");
//            logUrlInvalidError(customCampaignEndpoint, fallbackUrl);
//            return fallbackUrl;
//        }
    }

    public final String getActivateUrl() {
//        String customUrl = (customActivateDomain + customActivateEndpoint);
        String customUrl = (customActivateEndpoint);
        if (!isUrlValid(customUrl)) {
            String fallbackUrl = DEFAULT_DECISION_DOMAIN + DEFAULT_ACTIVATE_ENDPOINT;
            logUrlInvalidError(customUrl, fallbackUrl);
            return fallbackUrl;
        }
        return  customUrl;

//        if (isUrlValid(customActivateEndpoint))
//            return customActivateEndpoint;
//        else {
//            String fallbackUrl = DEFAULT_DECISION_DOMAIN + DEFAULT_ACTIVATE_ENDPOINT;
//            logUrlInvalidError(customActivateEndpoint, fallbackUrl);
//            return fallbackUrl;
//        }
    }

    public final String getEventUrl() {
//        String customUrl = (customEventDomain + customEventEndpoint);
        String customUrl = (customEventEndpoint);
        if (!isUrlValid(customUrl)) {
            String fallbackUrl = DEFAULT_DECISION_DOMAIN + DEFAULT_ACTIVATE_ENDPOINT;
            logUrlInvalidError(customUrl, fallbackUrl);
            return fallbackUrl;
        }
        return customUrl;
//        if (isUrlValid(customEventEndpoint))
//            return customEventEndpoint;
//        else {
//            String fallbackUrl = String.format(DEFAULT_DECISION_DOMAIN + DEFAULT_EVENTS_ENDPOINT, envId);
//            logUrlInvalidError(customCampaignEndpoint, fallbackUrl);
//            return fallbackUrl;
//        }
    }

    public final String getBucketingUrl() {
//        String customUrl = customBucketingDomain + customBucketingEndpoint;
        String customUrl = customBucketingEndpoint;
        if (!isUrlValid(customUrl)) {
            String fallbackUrl = DEFAULT_BUCKETING_DOMAIN + DEFAULT_BUCKETING_ENDPOINT;
            logUrlInvalidError(customUrl, fallbackUrl);
            return fallbackUrl;
        }
        return customUrl;
//        if (isUrlValid(customBucketingEndpoint))
//            return customBucketingEndpoint;
//        else {
//            String fallbackUrl = String.format(DEFAULT_BUCKETING_DOMAIN + DEFAULT_BUCKETING_ENDPOINT, envId);
//            logUrlInvalidError(customBucketingEndpoint, fallbackUrl);
//            return fallbackUrl;
//        }

    }

    public final String getTrackingUrl() {
//        String customUrl = (customTrackingDomain + customTrackingEndpoint);
        String customUrl = (customTrackingEndpoint);
        if (!isUrlValid(customUrl)) {
            String fallbackUrl = DEFAULT_TRACKING_DOMAIN;
            logUrlInvalidError(customUrl, fallbackUrl);
            return fallbackUrl;
        }
        return customUrl;
//        if (isUrlValid(customTrackingEndpoint))
//            return customTrackingEndpoint;
//        else {
//            String fallbackUrl = DEFAULT_TRACKING_DOMAIN;
//            logUrlInvalidError(customTrackingEndpoint, fallbackUrl);
//            return fallbackUrl;
//        }
    }

//    public final String getCampaignUrl(Boolean consent) {
//        String url = customDecisionDomain + String.format(CAMPAIGNS_ROUTE, envId) + (!consent ? CONTEXT_PARAM : "");
//        String overriddenUrl = overrideCampaignUrl(url);
//        return isUrlValid(overriddenUrl, url) ? overriddenUrl : url;
//    }
//
//    public final String getActivateUrl() {
//        return customDecisionDomain + ACTIVATE_ROUTE;
//    }
//
//    public final String getEventUrl() {
//        return customDecisionDomain + String.format(EVENTS_ROUTE, envId);
//    }
//
//    public final String getBucketingUrl() {
//        return customBucketingDomain + String.format(BUCKETING_ROUTE, envId);
//    }
//
//    public final String getTrackingUrl() {
//        return customTrackingDomain;
//    }

    @Override
    public String overrideCampaignUrl(String url) {
        return url;
    }

    private static Boolean isUrlValid(String url) {
        try {
            new URL(url);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    private static void logUrlInvalidError(String url, String fallbackUrl) {
        FlagshipLogManager.log(FlagshipLogManager.Tag.INITIALIZATION, LogManager.Level.ERROR,
                String.format(FlagshipConstants.Errors.CUSTOM_SERVICE_ENDPOINT_URL_ERROR, url, fallbackUrl));
    }


    public static class  Builder {

        private final FlagshipServiceEndpoints customService = new FlagshipServiceEndpoints();


        public  Builder withCampaignServiceUrl(String endpoint) {
            if (isUrlValid(endpoint))
                customService.customCampaignEndpoint = endpoint;
            return this;
        }

        public Builder withActivateServiceUrl(String endpoint) {
            if (isUrlValid(endpoint))
                customService.customActivateEndpoint = endpoint;
            return this;
        }

        public Builder withEventServiceUrl(String endpoint) {
            if (isUrlValid(endpoint))
                customService.customEventEndpoint = endpoint;
            return this;
        }

        public Builder withBucketingServiceUrl(String endpoint) {
            if (isUrlValid(endpoint))
                customService.customBucketingEndpoint = endpoint;
            return this;
        }

        public Builder withTrackingServiceUrl(String endpoint) {
            if (isUrlValid(endpoint))
                customService.customTrackingEndpoint = endpoint;
            return this;
        }

//        public Builder withServiceDomain(String domain) {
//            if (isUrlValid(domain)) {
//                customService.customCampaignDomain = domain;
//                customService.customActivateDomain = domain;
//                customService.customEventDomain = domain;
//                customService.customBucketingDomain = domain;
//                customService.customTrackingDomain = domain;
//            }
//            return this;
//        }
//
//
//       public  Builder withCampaignServiceUrl(String domain, String endpoint) {
//            if (isUrlValid(domain))
//                customService.customCampaignDomain = domain;
//           if (endpoint != null)
//                customService.customCampaignEndpoint = endpoint;
//           return this;
//        }
//
//        public Builder withActivateServiceUrl(String domain, String endpoint) {
//            if (isUrlValid(domain))
//                customService.customActivateDomain = domain;
//            if (endpoint != null)
//                customService.customActivateEndpoint = endpoint;
//            return this;
//        }
//
//        public Builder withEventServiceUrl(String domain, String endpoint) {
//            if (isUrlValid(domain))
//                customService.customEventDomain = domain;
//            if (endpoint != null)
//                customService.customEventEndpoint = endpoint;
//            return this;
//        }
//
//        public Builder withBucketingServiceUrl(String domain, String endpoint) {
//            if (isUrlValid(domain))
//                customService.customBucketingDomain = domain;
//            if (endpoint != null)
//                customService.customBucketingEndpoint = endpoint;
//            return this;
//        }
//
//        public Builder withTrackingServiceUrl(String domain, String endpoint) {
//            if (isUrlValid(domain))
//                customService.customTrackingDomain = domain;
//            if (endpoint != null)
//                customService.customTrackingEndpoint = endpoint;
//            return this;
//        }

        //________________________

//        public Builder withDecisionServiceDomain(String url) {
//            if (url.endsWith("/"))
//                url = url.substring(0, url.length() - 1);
//            if (isUrlValid(url))
//                customService.customDecisionDomain = url;
//            else
//                logUrlInvalidError(url, DEFAULT_DECISION_DOMAIN);
//            return this;
//        }
//
//        public Builder withBucketingServiceDomain(String url) {
//            if (url.endsWith("/"))
//                url = url.substring(0, url.length() - 1);
//            if (isUrlValid(url))
//                customService.customBucketingDomain = url;
//            else
//                logUrlInvalidError(url, DEFAULT_BUCKETING_DOMAIN);
//            return this;
//        }
//
//        public Builder withTrackingServiceDomain(String url) {
//            if (url.endsWith("/"))
//                url = url.substring(0, url.length() - 1);
//            if (isUrlValid(url))
//                customService.customTrackingDomain = url;
//            else
//                logUrlInvalidError(url, DEFAULT_TRACKING_DOMAIN);
//            return this;
//        }

        public FlagshipServiceEndpoints build() {
            return customService;
        }
    }
}
