package hope.common.spring.api;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import static java.nio.charset.StandardCharsets.UTF_8;

public abstract class ApiMetaResource implements InitializingBean {

  protected static final String EMPTY = "{}";
  protected final ApplicationContext applicationContext;
  protected Logger logger = LoggerFactory.getLogger(ApiMetaResource.class);
  protected String applicationName;

  protected String errors;
  protected String authorities;
  protected String versions;
  protected String dictionaries;

  protected ApiMetaResource(ApplicationContext applicationContext) {
    this.applicationContext = applicationContext;
  }

  @Override
  public void afterPropertiesSet() {
    applicationName = applicationContext.getApplicationName();
    if (!StringUtils.hasText(applicationName)) {

      applicationName = applicationContext.getEnvironment().getProperty("spring.application.name");
    }
  }

  private String safeLoadMeta(String name) {
    String path = applicationName + "/hope-domain-" + name + ".json";
    try {
      return StreamUtils.copyToString(
          getClass().getClassLoader().getResource(path).openStream(), UTF_8);
    } catch (Throwable throwable) {
      logger.warn("FAIL_LOAD " + path, throwable);
      return EMPTY;
    }
  }

  protected String safeApiByVersion(final String version){
    String path = applicationName + "/swagger/"+version+".json";
    try {
      return StreamUtils.copyToString(
              getClass().getClassLoader().getResource(path).openStream(), UTF_8);
    } catch (Throwable throwable) {
      logger.warn("FAIL_LOAD_API_BY_VERSION " + path, throwable);
      return EMPTY;
    }
  }

  protected String safeLoadVersions() {
    String path = applicationName + "/swagger/version.json";
    try {
      return StreamUtils.copyToString(
          getClass().getClassLoader().getResource(path).openStream(), UTF_8);
    } catch (Throwable throwable) {
      logger.warn("FAIL_LOAD " + path, throwable);
      return EMPTY;
    }
  }

  protected String dictionaries() {
    if (dictionaries == null) {
      dictionaries = safeLoadMeta("dictionaries");
    }
    return dictionaries;
  }

  protected String errors() {
    if (errors == null) {
      errors = safeLoadMeta("errors");
    }
    return errors;
  }

  protected String authorities() {
    if (authorities == null) {
      authorities = safeLoadMeta("authorities");
    }
    return authorities;
  }
  protected String versions() {
    if (versions == null) {
      versions = safeLoadVersions();
    }
    return versions;
  }
}
