package hope.common.spring.api.provider;

import static java.nio.charset.StandardCharsets.UTF_8;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import hope.common.api.exceptions.HopeErrorDetailException;
import hope.common.spring.api.Constants;
import hope.common.wire.errors.OASErrorsEnum;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

/**
 * this is load from the resources `resources/swagger/{application-name}/1.0.9-RELEASE.json` <code>
 *     "x-hope-project" : {
 *     "artifact" : {
 *       "groupId" : "com.bigger.user",
 *       "artifactId" : "it-demo-user-proto",
 *       "version" : "1.1.7-RELEASE"
 *     },
 *     "name" : "it-demo-user-proto",
 *     "description" : "用户模块",
 *     "application" : "it-demo-user",
 *     "packageName" : "com.bigger.user",
 *     "type" : "PROTO",
 *     "domain" : "user",
 *     "identifier" : "it-demo-user-proto",
 *     "status" : {
 *       "deprecated" : false,
 *       "createdTimestamp" : "2020-12-12",
 *       "updatedTimestamp" : "2022-10-19",
 *       "updatedBy" : "jake"
 *     }
 *   }
 * </code>
 */
public class ResourcesApiProvider implements ApiProvider, Constants {

  protected final Logger logger = LoggerFactory.getLogger(ResourcesApiProvider.class);

  protected final String applicationName;

  protected final String version;
  protected final boolean latest;
  protected final ObjectMapper objectMapper = new ObjectMapper();

  public ResourcesApiProvider(
      final String applicationName, final String version, final boolean latest) {
    this.applicationName = applicationName;

    this.version = version;
    this.latest = latest;
  }

  protected static String relativeJsonPath(final String applicationName, final String name) {
    return applicationName + "/" + RESOURCES_PATH + name + ".json";
  }

  @Override
  public String api() {

    String calculateVersion = version;
    if (!StringUtils.hasText(version)) {
      calculateVersion = loadLatestVersion();
    }

    String path = relativeJsonPath(applicationName, calculateVersion);

    try {
      final String content =
          StreamUtils.copyToString(
              getClass().getClassLoader().getResource(path).openStream(), UTF_8);
      return content;
    } catch (Throwable e) {
      throw HopeErrorDetailException.builder()
          .error(OASErrorsEnum.FAIL_READ_API_JSON)
          .setMessage(
              "fail read the version resource for application: ["
                  + applicationName
                  + "]  path: "
                  + path)
          .setCause(e)
          .build();
    }
  }

  @Override
  public String apiWithVersion(String version) {
    return apiWithVersion(version);
  }

  protected String loadLatestVersion() {

    final String versionPath = relativeJsonPath(applicationName, "version");
    try {
      final URL url = getClass().getClassLoader().getResource(versionPath);
      logger.debug("PROCESS_VERSION {}", url);
      JsonNode root = objectMapper.readTree(url);
      return pickLatestVersion(root);
    } catch (Throwable e) {
      throw HopeErrorDetailException.builder()
          .error(OASErrorsEnum.FAIL_READ_API_VERSION)
          .setMessage(
              "fail read the version resource for application: ["
                  + applicationName
                  + "]  path: "
                  + versionPath)
          .setCause(e)
          .build();
    }
  }

  /**
   * <code>
   *     {
   *       "current" : {
   *         "version" : "1.1.7-RELEASE",
   *         "createdTimestamp" : "2022-12-06",
   *         "updateTimestamp" : "2022-12-06"
   *       }
   *     }
   * </code>
   *
   * @param jsonNode
   * @return
   */
  protected String pickLatestVersion(final JsonNode jsonNode) {

    final JsonNode current = jsonNode.get("current");
    if (current != null) {
      return current.get("version").textValue();
    }
    return null;
  }
}
