package it.hope.plugin.wire;

import static it.hope.plugin.wire.HopeWireExtension.LAST_IT_PROTO_VERSION;

import it.hope.plugin.wire.tasks.WireCleanTask;
import it.hope.plugin.wire.tasks.WireGeneratorTask;
import it.hope.plugin.wire.tasks.WirePublishTask;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;
import org.gradle.api.Action;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.Task;
import org.gradle.api.artifacts.Configuration;
import org.gradle.api.file.FileCollection;
import org.gradle.api.file.FileTreeElement;
import org.gradle.api.plugins.AppliedPlugin;
import org.gradle.api.plugins.JavaPluginExtension;
import org.gradle.api.plugins.ObjectConfigurationAction;
import org.gradle.api.specs.Spec;
import org.gradle.api.tasks.SourceSet;
import org.gradle.api.tasks.bundling.Jar;
import org.gradle.api.tasks.compile.JavaCompile;

public class HopeWirePlugin implements Plugin<Project> {

  protected static final String WIRE_GRADLE_SCRIPT = "/wire-shared.gradle";
  static final Set<String> dependTasks =
      new HashSet<>() {
        {
          add("classes");
          add("compileJava");
          add("compileTestJava");
          add("testClasses");
        }
      };
  protected HopeWireExtension hopeWireExtension;

  protected HopeServerExtension hopeServerExtension;

  @Override
  public void apply(Project project) {

    // This by inject let the Gradle context do all those stuff
    hopeWireExtension = project.getExtensions().create("hopeWire", HopeWireExtension.class);

    hopeServerExtension = project.getExtensions().create("hopeServer", HopeServerExtension.class);

    final Action<Project> _action = self -> afterEvaluate(self);
    project.afterEvaluate(_action);
  }

  private void afterEvaluate(final Project project) {

    final Action<ObjectConfigurationAction> configurationAction =
        new Action<>() {
          @Override
          public void execute(ObjectConfigurationAction objectConfigurationAction) {
            objectConfigurationAction.from(getClass().getResource(WIRE_GRADLE_SCRIPT));
          }
        };

    final Action<AppliedPlugin> javaApplied = appliedPlugin -> project.apply(configurationAction);
    project.getPluginManager().withPlugin("java", javaApplied);
    // after all done we will bind the last target task:

    if (hopeWireExtension.getDisable().getOrElse(Boolean.FALSE)) {
      project.getLogger().warn("WIRE_PLUGIN_DISABLE_NO_MORE_ACTION");
      return;
    }

    buildWireStandAloneTask(project);
    injectJarTask(project);
    configureUtf8Encoding(project);
  }

  private void injectJarTask(final Project project) {

    // before JAR, we need to compile the `hopeWireUp`
    // any we need to exclude the proto generated class
    final boolean keepProto = hopeWireExtension.getKeepProto().getOrElse(Boolean.FALSE);
    project
        .getLogger()
        .warn("PROTOC_CLASS_WILL_{} {}", keepProto ? "INCLUDED" : "EXCLUDED", project.getName());
    if (!keepProto) {
      project
          .getTasks()
          .withType(
              Jar.class,
              jar -> {
                final Spec<FileTreeElement> spec =
                    element -> {
                      if (qualifyName(element.getName())) {
                        if (element.getPath().contains("/proto")) {
                          project.getLogger().debug("EXCLUDE_PROTO_PATH {}", element.getPath());
                          return true;
                        }
                      }
                      return false;
                    };

                jar.exclude(spec);
              });
    }
  }

  /**
   *
   *
   * <pre>
   *     we refer to the
   *     https://github.com/swagger-api/swagger-core/blob/master/modules/swagger-gradle-plugin/README.md
   * </pre>
   *
   * @param project
   */
  private void buildWireStandAloneTask(final Project project) {

    final Configuration config =
        project.getConfigurations().create("wireStandalone").setVisible(false);

    config.defaultDependencies(
        dependencies -> {
          // and the it-wire  in class path as we will generate the class and then build it
          final String wireVersion =
              "com.apihug:it-wire:"
                  + hopeWireExtension.getPluginMainVersion().getOrElse(LAST_IT_PROTO_VERSION);
          project.getLogger().warn("it-wire runtime: {}", wireVersion);

          if (hopeWireExtension.getLocal().getOrElse(Boolean.FALSE)) {
            project.getLogger().warn("[com.apihug:it-wire] applied no need to add again");
          } else {
            project.getLogger().warn("add dependency of the [com.apihug:it-wire]: {}", wireVersion);
            dependencies.add(project.getDependencies().create(wireVersion));
          }
          // runtime logger
          // for the runtime debug this is another way no the gradle plugin logger managed
          dependencies.add(
              project.getDependencies().create("ch.qos.logback:logback-classic:1.4.5"));
        });

    final Callable<FileCollection> classpath =
        () ->
            javaPluginExtension(project)
                .getSourceSets()
                .findByName(SourceSet.MAIN_SOURCE_SET_NAME)
                .getRuntimeClasspath()
                .filter(new JarTypeFileSpec());

    WireGeneratorTask task = project.getTasks().create("wire", WireGeneratorTask.class);
    task.setGroup("build");
    task.setBuildClasspath(config);
    task.setClasspath(classpath);
    task.setExtension(hopeWireExtension);

    for (String dependTask : dependTasks) {
      try {
        if (project.getTasks().findByPath(dependTask) != null) {
          task.dependsOn(dependTask);
        }
      } catch (Exception e) {
        project.getLogger().warn("Exception in task dependencies: " + e.getMessage(), e);
      }
    }

    // This should always run after the 'build' task has completed, the last
    final Task build = project.getTasks().findByPath("build");
    if (build != null) {
      // before build must wire
      build.dependsOn(task);
    }

    WireCleanTask cleanTask = project.getTasks().create("wireClean", WireCleanTask.class);
    cleanTask.setGroup("build");

    // TODO test it carefully
    // After wire we will do the publishing
    WirePublishTask publishTask = project.getTasks().create("wirePublish", WirePublishTask.class);
    publishTask.setExtension(hopeServerExtension);
    // publishTask.dependsOn(task);
    publishTask.setGroup("publishing");
  }

  private boolean qualifyName(final String name) {
    if (name != null && !name.isEmpty()) {
      return name.endsWith(".java") || name.endsWith(".class");
    }
    return false;
  }

  private JavaPluginExtension javaPluginExtension(Project project) {
    return project.getExtensions().getByType(JavaPluginExtension.class);
  }

  private void configureUtf8Encoding(Project evaluatedProject) {
    evaluatedProject
        .getTasks()
        .withType(JavaCompile.class)
        .configureEach(this::configureUtf8Encoding);
  }

  private void configureUtf8Encoding(JavaCompile compile) {
    if (compile.getOptions().getEncoding() == null) {
      compile.getOptions().setEncoding("UTF-8");
    }
  }
}
