package it.hope.plugin.wire;

import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.file.ProjectLayout;
import org.gradle.api.model.ObjectFactory;
import org.gradle.api.provider.Property;

import javax.inject.Inject;

/**
 * <code>
 *     This wire proto plugin runtime depend on google's protobuf gradle plugin:
 *
 *     com.google.protobuf:protobuf-gradle-plugin:0.9.1
 *
 *     proto compile version:
 *
 *     protoc {
 *         artifact = 'com.google.protobuf:protoc:3.21.7'
 *     }
 *
 *    grpc compile version:
 *
 *    grpc {
 *       artifact = 'io.grpc:protoc-gen-grpc-java:1.50.0'
 *    }
 *
 *    ------------------
 *    All pkg under the com.google.protobuf will go with the #PROTOC_VERSION
 *
 *    com.google.protobuf
 *        protobuf-java:
 *            Core Protocol Buffers library. Protocol Buffers are a way of encoding structured data in an efficient yet extensible format.
 *        protobuf-java-util
 *            Utilities for Protocol Buffers
 *           protobuf-javalite:
 *            Lite version of Protocol Buffers library. This version is optimized for code size, but does not guarantee API/ABI stability.
 *
 *   ------------------
 *   All pkg under the io.grpc will go with the #PROTOC_GRPC_VERSION
 *
 *   io.grpc
 *        grpc-stub:
 *            Stub classes for GRPC which are exposed to developers and provides type-safe bindings.
 *        grpc-protobuf:
 *            gRPC: Protobuf
 *        grpc-netty：
 *            gRPC: Netty
 *        grpc-api:
 *            gRPC: API
 *        grpc-core:
 *            gRPC: Core
 *        grpc-context
 *        grpc-netty-shaded
 *        grpc-protobuf-lite
 *        grpc-services
 *        grpc-auth
 *
 * </code> those are last compatible version:
 *
 * @see #PROTOC_VERSION
 * @see #PROTOC_GRPC_VERSION
 */
public class HopeWireExtension {

  public static final String PROTOC_VERSION = "3.21.7";
  public static final String PROTOC_GRPC_VERSION = "1.50.0";
  public static final String SWAGGER_OPEN_API_VERSION = "2.2.3";

  public static final String VALIDATION_VERSION = "2.0.1.Final";

  /** Developer please help to maintain this last one */
  public static final String LAST_IT_PROTO_VERSION = "0.9.9-RELEASE";

  /** if the customized wire plugin assistant main class is different, set here */
  protected final Property<String> pluginMainClass;

  protected final Property<Boolean> keepProto;
  protected final Property<Boolean> verbose;
  protected final Property<Boolean> generatedVersion;
  protected final Property<Boolean> generatedTime;

  /** Exclude the wire source from the source version contorl like the .gitignore */
  protected final Property<Boolean> ignoreWireSource;

  /** if any customized wire plugin assistant version please set here */
  private final Property<String> pluginMainVersion;

  private final Property<String> grpcVersion;
  private final Property<String> protocVersion;

  /** if disable is true this plugin will not go live */
  private final Property<Boolean> disable;

  /** this for the local test only please do not touch it */
  private final Property<Boolean> local;

  /** debug only for dry run, just print every thing */
  private final Property<Boolean> debug;

  private final Property<Boolean> restrict;
  private final Property<String> swaggerVersion;
  private final Property<String> validationVersion;
  private final Property<Boolean> smock;

  /** Whether enable Kola test generator */
  private final Property<Boolean> kola;

  @Inject
  public HopeWireExtension(ProjectLayout layout, ObjectFactory objects) {

    grpcVersion = objects.property(String.class);
    protocVersion = objects.property(String.class);

    pluginMainClass = objects.property(String.class);

    pluginMainClass.set("hope.proto.wire.Main");
    keepProto = objects.property(Boolean.class);

    pluginMainVersion = objects.property(String.class);
    disable = objects.property(Boolean.class);

    local = objects.property(Boolean.class);

    disable.set(Boolean.FALSE);
    local.set(Boolean.FALSE);

    ignoreWireSource = objects.property(Boolean.class);
    ignoreWireSource.set(Boolean.FALSE);
    final String version = HopeWirePlugin.class.getPackage().getImplementationVersion();
    if (version != null && !version.isBlank()) {
      pluginMainVersion.set(version);
    } else {
      pluginMainVersion.set(LAST_IT_PROTO_VERSION);
    }

    grpcVersion.set(PROTOC_GRPC_VERSION);
    protocVersion.set(PROTOC_VERSION);

    keepProto.set(Boolean.TRUE);

    swaggerVersion = objects.property(String.class);
    swaggerVersion.set(SWAGGER_OPEN_API_VERSION);

    debug = objects.property(Boolean.class);
    debug.set(false);

    restrict = objects.property(Boolean.class);
    restrict.set(true);

    verbose = objects.property(Boolean.class);
    verbose.set(false);

    generatedVersion = objects.property(Boolean.class);
    generatedVersion.set(false);

    generatedTime = objects.property(Boolean.class);
    generatedTime.set(false);

    validationVersion = objects.property(String.class);
    validationVersion.set(VALIDATION_VERSION);

    smock = objects.property(Boolean.class);
    smock.set(false);

    kola = objects.property(Boolean.class);
    kola.set(true);
  }

  public Property<Boolean> getKola() {
    return kola;
  }

  public Property<String> getValidationVersion() {
    return validationVersion;
  }

  public Property<String> getGrpcVersion() {
    return grpcVersion;
  }

  public Property<String> getProtocVersion() {
    return protocVersion;
  }

  public Property<String> getPluginMainClass() {
    return pluginMainClass;
  }

  public Property<Boolean> getKeepProto() {
    return keepProto;
  }

  public Property<String> getPluginMainVersion() {
    return pluginMainVersion;
  }

  public Property<Boolean> getDisable() {
    return disable;
  }

  public Property<Boolean> getLocal() {
    return local;
  }

  public Property<Boolean> getDebug() {
    return debug;
  }

  public Property<Boolean> getRestrict() {
    return restrict;
  }

  public Property<String> getSwaggerVersion() {
    return swaggerVersion;
  }

  public Property<Boolean> getSmock() {
    return smock;
  }

  public Property<Boolean> getVerbose() {
    return verbose;
  }

  public Property<Boolean> getGeneratedVersion() {
    return generatedVersion;
  }

  public Property<Boolean> getGeneratedTime() {
    return generatedTime;
  }

  public Property<Boolean> getIgnoreWireSource() {
    return ignoreWireSource;
  }
}
