package it.hope.plugin.wire.tasks;

import static java.util.stream.Collectors.joining;

import it.hope.plugin.wire.HopeServerExtension;
import it.hope.plugin.wire.ZipUtils;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import org.gradle.api.DefaultTask;
import org.gradle.api.logging.Logging;
import org.gradle.api.tasks.TaskAction;
import org.gradle.work.DisableCachingByDefault;
import org.slf4j.Logger;

@DisableCachingByDefault(
    because = "Hope wire publish task will publish the wire proto to the server")
public class WirePublishTask extends DefaultTask {

  private static Logger logger = Logging.getLogger(WirePublishTask.class);

  protected HopeServerExtension extension;

  @TaskAction
  public void publish() {
    logger.warn("TRY_PUBLISH_WIRE_TO_SERVICE: ZIP->Publish need proxy + appKey + appSecret");
    // Zip the generated files
    // src/main/resources

    final String url = extension.getUrl().getOrElse(null);
    final String appKey = extension.getAppKey().getOrElse(null);
    final String appSecret = extension.getAppSecret().getOrElse(null);

    // TODO check url is legal, and key and secret is Set right?
    // FIXME or Get from the System env properties

    final String baseDir = getProject().getProjectDir().getAbsolutePath();

    final File resourcesDir = Path.of(baseDir, "src", "main", "resources").toFile();

    logger.warn("PREPARE_RESOURCE_FROM_LOCATION: {}", resourcesDir);

    if (!resourcesDir.exists()) {
      logger.warn("NO_RESOURCE_FOUND: {}", resourcesDir);
      return;
    }
    if (!resourcesDir.isDirectory()) {
      logger.warn("RESOURCE_IS_NOT_DIR: {}", resourcesDir);
      return;
    }
    final String projectName = getProject().getName();

    final Path path;
    try {
      path = ZipUtils.zip(resourcesDir, "metaFile");
      logger.warn("ZIP_RESOURCE_AT {}", path.toAbsolutePath());
      // Then we submit to our saas' service
    } catch (IOException e) {
      throw new RuntimeException("FAIL_ZIP_RESOURCE " + resourcesDir, e);
    }

    final Map<String, String> requestParams = new HashMap<>();

    final String version = extension.getVersion().getOrElse(null);
    if (version != null && !version.isBlank()) {
      requestParams.put("version", version);
    }

    final String name = extension.getName().getOrElse(null);
    if (name != null && !name.isBlank()) {
      requestParams.put("name", name);
    } else {
      requestParams.put("name", projectName);
    }

    requestParams.put("appKey", appKey);
    requestParams.put("appSecret", appSecret);

    // TODO collect runtime env
    //    2. OS
    //    3. User
    //    4. Java version
    //    5. Java vendor
    //    6. Git ?

    final String encodedURL =
        requestParams.keySet().stream()
            .map(key -> key + "=" + encodeValue(requestParams.get(key)))
            .collect(joining("&", url, ""));

    logger.warn("TRY_PUBLISH_WIRE_TO_SERVICE: {}", encodedURL);
    upload(path, encodedURL);
  }

  private String encodeValue(String value) {
    try {
      return URLEncoder.encode(value, StandardCharsets.UTF_8.toString());
    } catch (UnsupportedEncodingException e) {
      throw new RuntimeException(e);
    }
  }

  private void upload(final Path path, final String url) {

    // Those are since JAVA-11  things need to test it carefully before use it

    HttpClient client = HttpClient.newHttpClient();
    HttpRequest.Builder hb;
    try {
      // FIXME safe hole, how to encrypt it &  safe chanel no middle-attack; must sign it
      hb = HttpRequest.newBuilder(URI.create(url)).timeout(Duration.ofMinutes(3));
    } catch (Throwable e) {
      throw new RuntimeException(e);
    }

    try {

      multipleForm(hb, path);
      HttpResponse<String> responseOfString =
          client.send(hb.build(), HttpResponse.BodyHandlers.ofString());

      if (200 != responseOfString.statusCode()) {
        throw new RuntimeException("FAIL_UPLOAD " + path + " " + responseOfString.statusCode());
      }

      logger.debug("UPLOAD_SUCCESS: {} WITH_RESPONSE {}", path, responseOfString.body());
    } catch (IOException e) {
      throw new RuntimeException("IO_EXCEPTION " + path, e);
    } catch (InterruptedException e) {
      throw new RuntimeException("INTERRUPT " + path, e);
    }
  }

  public WirePublishTask setExtension(HopeServerExtension extension) {
    this.extension = extension;
    return this;
  }

  /**
   * refer to the {@code https://gitlab.com/-/snippets/2001189}
   *
   * @param builder
   * @param path
   * @return
   * @throws IOException
   */
  private HttpRequest.Builder multipleForm(HttpRequest.Builder builder, final Path path)
      throws IOException {

    final String boundary = new BigInteger(35, new Random()).toString();

    final byte[] separator =
        ("--" + boundary + "\r\nContent-Disposition: form-data; name=")
            .getBytes(StandardCharsets.UTF_8);
    var byteArrays = new ArrayList<byte[]>();
    byteArrays.add(separator);

    final String miniType = Files.probeContentType(path);

    byteArrays.add(
        ("\"metaFile\"; filename=\"metaFile\"\r\nContent-Type: " + miniType + "\r\n\r\n")
            .getBytes(StandardCharsets.UTF_8));
    byteArrays.add(Files.readAllBytes(path));
    byteArrays.add("\r\n".getBytes(StandardCharsets.UTF_8));
    byteArrays.add("--$boundary--".getBytes(StandardCharsets.UTF_8));

    builder
        .header("Content-Type", "multipart/form-data;boundary=" + boundary)
        .POST(HttpRequest.BodyPublishers.ofByteArrays(byteArrays));

    return builder;
  }
}
