package it.hope.plugin.wire.tasks;

import it.hope.plugin.wire.HopeWireExtension;
import it.hope.plugin.wire.json.JSONObject;
import java.io.File;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.Callable;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.gradle.api.DefaultTask;
import org.gradle.api.GradleException;
import org.gradle.api.Project;
import org.gradle.api.file.FileCollection;
import org.gradle.api.logging.Logging;
import org.gradle.api.tasks.*;
import org.gradle.work.DisableCachingByDefault;
import org.slf4j.Logger;

/**
 * @see JavaExec
 */
@SuppressWarnings("Duplicates")
@DisableCachingByDefault(because = "Hope wire main task will do proto collect and generate")
public class WireGeneratorTask extends DefaultTask {

  private static final Logger logger = Logging.getLogger(WireGeneratorTask.class);

  private Iterable<File> buildClasspath;

  private Callable<FileCollection> classpath;

  @Internal private String mainClassName = "hope.proto.wire.Main";

  private HopeWireExtension extension;

  protected static Method pick(final String mainClassName, final ClassLoader classLoader)
      throws ClassNotFoundException, NoSuchMethodException {

    Class clz = classLoader.loadClass(mainClassName);
    Method mainMethod = clz.getDeclaredMethod("main", String[].class);
    mainMethod.setAccessible(true);
    return mainMethod;
  }

  public static void verifyWireMeta(final Project project) {
    File meta =
        Paths.get(
                project.getProjectDir().getAbsolutePath(),
                "src",
                "main",
                "resources",
                "hope-wire.json")
            .toFile();

    final boolean hasMeta = meta.exists() && meta.canRead();

    if (!hasMeta) {
      throw new IllegalStateException(
          "Since [0.1.2-RELEASE] you need place project meta file under the "
              + meta.getAbsolutePath());
    }
  }

  public static void writeConfig(
      final Project project, final Path path, final HopeWireExtension extension) throws Exception {
    final JSONObject wireConfiguration = new JSONObject();

    /**
     * <code>
     *     "artifact": {
     *     "groupId": "com.bigger.pet",
     *     "artifactId": "bigger-pet-proto",
     *     "version": "1.1.1"
     *   }
     * </code>
     */
    // artifact part
    JSONObject artifact = new JSONObject();
    artifact.put("groupId", project.getGroup().toString());
    artifact.put("artifactId", project.getName());
    artifact.put("version", project.getVersion().toString());
    wireConfiguration.put("artifact", artifact);
    /**
     * <code>
     *    "name": "bigger-pet-proto",
     *   "description": "bigger company pet store example",
     *   "application": "bigger-pet",
     *   "packageName": "com.bigger.pet",
     *   "type": "PROTO",
     *   "domain": "order",
     *   "identifier": "bigger-pet-proto",
     * </code>
     */
    wireConfiguration.put("name", project.getName());
    wireConfiguration.put("description", project.getDescription());

    /**
     * those from the proto so not do here! <code>
     *     "status": {
     *     "deprecated": false,
     *     "deprecatedBy": "jake",
     *     "deprecatedReason": null,
     *     "deprecatedSince": null,
     *     "deprecatedUpdatedTimestamp": null,
     *     "createdTimestamp": "2020-12-12",
     *     "updatedTimestamp": "2022-10-10",
     *     "updatedBy": "jake"
     *   }
     * </code>
     */

    // ------------------------------
    // Builder part
    /**
     * <code>
     *      "builder": {
     *     "dir": "/var/tmp/bigger/pet",
     *     "buildDir": "/var/tmp/bigger/build",
     *     "parent": "bigger-picture",
     *     "properties": {
     *       "key1": "value1"
     *     }
     *   }
     * </code>
     */
    JSONObject builder = new JSONObject();
    builder.put("dir", project.getProjectDir().getAbsolutePath());
    builder.put("buildDir", project.getBuildDir().getAbsolutePath());
    builder.put("pluginVersion", extension.getPluginMainVersion().getOrElse("NA"));

    if (project.getParent() != null) {
      builder.put("parent", pickParent(project.getParent(), null));
    }
    wireConfiguration.put("builder", builder);

    /**
     * <code>
     *     "wire": {
     *     "strict": false,
     *     "debug": false,
     *     "protoGenDir": "protoGen",
     *     "wireGenDir": "wireGen"
     *   }
     * </code>
     */
    JSONObject wire = new JSONObject();
    wire.put("debug", extension.getDebug().getOrElse(Boolean.FALSE).booleanValue());
    wire.put("restrict", extension.getRestrict().getOrElse(Boolean.TRUE).booleanValue());

    wire.put("protoGenDir", "protoGen");
    wire.put("wireGenDir", "wireGen");
    wire.put("keepProto", extension.getKeepProto().getOrElse(Boolean.TRUE).booleanValue());

    final String wirePath =
        project
            .getProjectDir()
            .toPath()
            .resolve("src")
            .resolve("main")
            .resolve("wire")
            .toAbsolutePath()
            .toString();

    wire.put("wirePath", wirePath);
    wireConfiguration.put("wire", wire);

    final String config = wireConfiguration.toString(2);

    Files.write(path, config.getBytes(StandardCharsets.UTF_8));

    if (extension.getVerbose().getOrElse(false)) {
      logger.debug("\n\nConfig @ [{}] \n{}\n", path.toAbsolutePath(), config);
    }
  }

  public static String safeGetProperty(final String key) {
    try {
      return System.getProperty(key);
    } catch (Throwable throwable) {

    }
    return null;
  }

  protected static String pickParent(final Project project, final String prefix) {
    if (project != null && project.getParent() != null) {
      // project(":it-full-example:it-demo-inventory-proto")
      String newPrefix;
      if (prefix == null || prefix.isBlank()) {
        newPrefix = project.getParent().getName();
      } else {
        newPrefix = project.getParent().getName() + ":" + prefix;
      }
      return pickParent(project.getParent().getParent(), newPrefix);
    } else {
      return prefix;
    }
  }

  public WireGeneratorTask setClasspath(Callable<FileCollection> classpath) {
    this.classpath = classpath;
    return this;
  }

  @Classpath
  @Optional
  public Iterable<File> getBuildClasspath() {
    return buildClasspath;
  }

  public void setBuildClasspath(Iterable<File> buildClasspath) {
    this.buildClasspath = buildClasspath;
  }

  public String getMainClassName() {
    return mainClassName;
  }

  public WireGeneratorTask setMainClassName(String mainClassName) {
    this.mainClassName = mainClassName;
    return this;
  }

  @TaskAction
  public void doWire() throws GradleException {

    verifyWireMeta(getProject());

    final boolean verbose = extension.getVerbose().getOrElse(false);

    logger.info("Resolving OpenAPI specification..");

    Stream<URL> classpathStream = Stream.empty();

    if (classpath != null) {

      try {
        classpathStream =
            StreamSupport.stream(classpath.call().spliterator(), false)
                .map(
                    f -> {
                      try {
                        return f.toURI().toURL();
                      } catch (MalformedURLException e) {
                        throw new GradleException(
                            String.format(
                                "Could not create classpath for annotations task %s.", getName()),
                            e);
                      }
                    });
      } catch (Exception e) {
        logger.error("FAIL_GET_CLASS_PATH_RUN_TIME {}", e);
        throw new RuntimeException(e);
      }
    }

    Stream<URL> buildClasspathStream = Stream.empty();

    if (getBuildClasspath() != null) {
      buildClasspathStream =
          StreamSupport.stream(getBuildClasspath().spliterator(), false)
              .map(
                  f -> {
                    try {
                      return f.toURI().toURL();
                    } catch (MalformedURLException e) {
                      throw new GradleException(
                          String.format(
                              "Could not create classpath for annotations task %s.", getName()),
                          e);
                    }
                  });
    }

    final URL[] urls =
        Stream.concat(classpathStream, buildClasspathStream).distinct().toArray(URL[]::new);

    for (final URL url : urls) {
      if (verbose) {
        logger.debug("WIRE_RUN_TIME_CLASS_PATH: {}", url);
      }
    }

    final ClassLoader classLoader = new URLClassLoader(urls);

    try {

      final Method mainMethod = pick(mainClassName, classLoader);
      final Path path = Files.createTempFile("it-plugin-wire-", ".json");
      writeConfig(getProject(), path, extension);

      if (extension.getSmock().getOrElse(Boolean.FALSE)) {
        logger.warn("SMOCK_VERIFY_SKIP_FURTHER_ACTION");
        return;
      }

      String forceOverwrite = safeGetProperty("forceOverwrite");

      final String[] converter =
          new String[] {
            "-Dinput.args=" + path.toAbsolutePath(),
            "-DbuildVerbose=" + extension.getVerbose().getOrElse(false),
            "-DgeneratedVersion=" + extension.getGeneratedVersion().getOrElse(false),
            "-DgeneratedTime=" + extension.getGeneratedTime().getOrElse(false),
            "-DignoreWireSource=" + extension.getIgnoreWireSource().getOrElse(false),
            "-Dkola=" + extension.getKola().getOrElse(false),
            "-DforceOverwrite=" + "true".equals(forceOverwrite),
          };

      mainMethod.invoke(null, (Object) converter);

    } catch (Throwable throwable) {
      logger.error("fail wire generator", throwable);
      // FIXME release the lock ?
      throw new RuntimeException(throwable);
    }
  }

  public void setExtension(HopeWireExtension extension) {
    this.extension = extension;
  }
}
