/*
* Copyright 2015 Axibase Corporation or its affiliates. All Rights Reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License").
* You may not use this file except in compliance with the License.
* A copy of the License is located at
*
* https://www.axibase.com/atsd/axibase-apache-2.0.pdf
*
* or in the "license" file accompanying this file. This file is distributed
* on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
* express or implied. See the License for the specific language governing
* permissions and limitations under the License.
*/
package com.axibase.tsd.client;

import com.axibase.tsd.model.data.Alert;
import com.axibase.tsd.model.data.AlertHistory;
import com.axibase.tsd.model.data.Property;
import com.axibase.tsd.model.data.Severity;
import com.axibase.tsd.model.data.command.*;
import com.axibase.tsd.model.data.series.GetSeriesBatchResult;
import com.axibase.tsd.model.data.series.GetSeriesResult;
import com.axibase.tsd.query.Query;
import com.axibase.tsd.query.QueryPart;

import java.util.Arrays;
import java.util.List;

import static com.axibase.tsd.client.RequestProcessor.patch;
import static com.axibase.tsd.client.RequestProcessor.post;
import static com.axibase.tsd.util.AtsdUtil.check;

/**
 * Provides high-level API to retrieve and update ATSD Data Objects (time-series, alerts, properties).
 *
 * @author Nikolay Malevanny.
 */
public class DataService {
    public static final SeriesCommandPreparer LAST_PREPARER = new LastPreparer();

    private HttpClientManager httpClientManager;

    public DataService() {
    }

    public DataService(HttpClientManager httpClientManager) {
        this.httpClientManager = httpClientManager;
    }

    public void setHttpClientManager(HttpClientManager httpClientManager) {
        this.httpClientManager = httpClientManager;
    }

    /**
     * @param seriesQueries queries with details, each query property overrides common one in the request parameters
     * @return list of {@code GetSeriesResult}
     */
    public List<GetSeriesResult> retrieveSeries(GetSeriesCommand... seriesQueries) {
        QueryPart<GetSeriesBatchResult> query = new Query<GetSeriesBatchResult>("series");
        GetSeriesBatchResult seriesBatchResult = httpClientManager.requestData(GetSeriesBatchResult.class, query,
                post(new GetSeriesBatchCommand(Arrays.asList(seriesQueries))));
        return seriesBatchResult.getSeriesResults();
    }

    public List<GetSeriesResult> retrieveSeries(SeriesCommandPreparer preparer, GetSeriesCommand... seriesQueries) {
        if (preparer != null) {
            for (GetSeriesCommand seriesQuery : seriesQueries) {
                preparer.prepare(seriesQuery);
            }
        }
        return retrieveSeries(seriesQueries);
    }

    /**
     * @param addSeriesCommands commands that contains time-series which are added
     * @return true if success
     */
    public boolean addSeries(AddSeriesCommand... addSeriesCommands) {
        QueryPart<GetSeriesResult> query = new Query<GetSeriesResult>("series")
                .path("insert");
        return httpClientManager.updateData(query,
                post(Arrays.asList(addSeriesCommands)));
    }

    /**
     * @param entityName entity name
     * @param data CSV as String
     * @param tagNamesAndValues entity tags
     * @return true if success
     */
    public boolean addSeriesCsv(String entityName, String data, String... tagNamesAndValues) {
        check(entityName, "Entity name is empty");
        check(data, "Data is empty");
        QueryPart<GetSeriesResult> query = new Query<GetSeriesResult>("series")
                .path("csv")
                .path(entityName);
        if (tagNamesAndValues != null) {
            if (tagNamesAndValues.length % 2 == 1) {
                throw new IllegalArgumentException("Tag without value");
            }
            for (int i = 0; i < tagNamesAndValues.length; i++) {
                query = query.param(tagNamesAndValues[i], tagNamesAndValues[++i]);
            }
        }
        return httpClientManager.updateData(query, data);
    }

    /**
     * @param seriesQueries queries with details
     * @return list of {@code GetSeriesResult}
     */
    public List<GetSeriesResult> retrieveLastSeries(GetSeriesCommand... seriesQueries) {
        return retrieveSeries(LAST_PREPARER, seriesQueries);
    }

    /**
     * @param getPropertiesCommand command with property filter parameters
     * @return list of {@code Property}
     */
    public List<Property> retrieveProperties(GetPropertiesCommand getPropertiesCommand) {
        QueryPart<Property> query = new Query<Property>("properties");
        return httpClientManager.requestDataList(Property.class, query,
                post(getPropertiesCommand));
    }

    /**
     * @param properties list of {@code Property} to add.
     * @return true if success
     */
    public boolean insertProperties(Property... properties) {
        QueryPart<Property> query = new Query<Property>("properties")
                .path("insert");
        return httpClientManager.updateData(query, post(Arrays.asList(properties)));
    }

    /**
     * @param batchPropertyCommands list of batch commands to mass update properties
     * @return true if success
     */
    public boolean batchUpdateProperties(BatchPropertyCommand... batchPropertyCommands) {
        QueryPart<Property> query = new Query<Property>("properties");
        return httpClientManager.updateData(query, patch(batchPropertyCommands));
    }

    /**
     * @param metricNames metric filter, multiple values allowed
     * @param entityNames entity filter, multiple values allowed
     * @param ruleNames rule filter, multiple values allowed
     * @param severities severity filter, multiple values allowed
     * @param minSeverity minimal severity filter
     * @return list of {@code Alert}
     */
    public List<Alert> retrieveAlerts(
            List<String> metricNames,
            List<String> entityNames,
            List<String> ruleNames,
            List<Severity> severities,
            Severity minSeverity) {
        QueryPart<Alert> query = new Query<Alert>("/alerts");
        if (minSeverity != null) {
            query.param("min-severity", minSeverity.getCode());
        }
        query = fillParams(query, "metric", metricNames);
        query = fillParams(query, "entity", entityNames);
        query = fillParams(query, "rule", ruleNames);
        if (severities != null) {
            for (Severity severity : severities) {
                query = query.param("severity", severity.getCode());
            }
        }
        return httpClientManager.requestDataList(Alert.class, query, null);
    }

    /**
     * @param getAlertHistoryCommand command with alert history selection details
     * @return list of  {@code AlertHistory}
     */
    public List<AlertHistory> retrieveAlertHistory(GetAlertHistoryCommand getAlertHistoryCommand) {
        QueryPart<AlertHistory> query = new Query<AlertHistory>("alerts")
                .path("history");
        return httpClientManager.requestDataList(AlertHistory.class, query,
                post(getAlertHistoryCommand));
    }

    private static  QueryPart<Alert> fillParams(QueryPart<Alert> query, String paramName, List<String> paramValueList) {
        if (paramValueList != null) {
            for (String paramValue : paramValueList) {
                query = query.param(paramName, paramValue);
            }
        }
        return query;
    }

    private static class LastPreparer implements SeriesCommandPreparer {
        @Override
        public void prepare(GetSeriesCommand command) {
            command.setLast(true);
        }
    }
}
