package blade.plugin.sql2o;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.sql.DataSource;

import org.sql2o.Connection;
import org.sql2o.Query;
import org.sql2o.Sql2o;

import blade.kit.CollectionKit;
import blade.kit.StringKit;
import blade.log.Logger;

/**
 * 实体对象基类
 *
 * @author	<a href="mailto:biezhi.me@gmail.com" target="_blank">biezhi</a>
 * @since	1.0
 */
public class Model {
	
	private static final Logger LOGGER = Logger.getLogger(Model.class);

    private static Sql2o sql2o; 
    
    private Class<? extends Model> model;
    
    public Model() {
    	this.model = this.getClass();
	}
    
    public Model(Class<? extends Model> clazz) {
		this.model = clazz;
	}
    
    /**
     * 设置sql2o实例
     * @param dataSource
     * @return
     */
    static Sql2o sql2o(DataSource dataSource){
    	sql2o = new Sql2o(dataSource);
    	return sql2o;
    }
    
    public static Model getModel(Class<? extends Model> clazz){
    	return new Model(clazz);
    }
    
    public Sql2o getSql2o(){
    	return sql2o;
    }
    
    /**
     * @return	返回表名称
     */
    public String getTable(){
    	return model.getAnnotation(Table.class).value();
    }
    
    /**
     * @return	返回表主键
     */
    public String getPK(){
    	return model.getAnnotation(Table.class).PK();
    }
    
    //	存储sql
    private String sql;
    
    //	存储排序
    private String orderby;
    
    //	存储操作,增删改查
    private DmlType dmlType;
    
    //	存储设置的参数,insert-update用
    private Map<String, Object> params;
    
    //	存储where条件，select-update-delete用
    private Map<String, Object> whereParams;

    // 大于
    private Map<String, Object> greaterParams;
    // 大于等于
    private Map<String, Object> greaterThanParams;
    // 小于
    private Map<String, Object> lessParams;
    // 小于等于
    private Map<String, Object> lessThanParams;
    // like
    private Map<String, Object> likeParams;
    
    private enum DmlType{
    	SELECT, COUNT, INSERT, UPDATE, DELETE
    }
    
    /**
     * @return	返回查询model对象，推荐方式
     */
    public Model select(){
    	this.sql = "select * from " + getTable();
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.SELECT;
    	return this;
    }
    
    /**
     * 自定义sql返回查询model对象
     * 
     * @param sql	要查询的sql语句
     * @return		返回查询model对象
     */
    public Model select(String sql){
    	this.sql = sql;
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.SELECT;
    	return this;
    }
	
    /**
     * @return	返回计算count
     */
    public Model count(){
    	this.sql  = "select count(1) from " + getTable();
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.COUNT;
    	return this;
    }
    
    /**
     * 自定义sql返回查询model对象
     * 
     * @param sql	要查询的sql语句
     * @return		返回查询model对象
     */
    public Model count(String sql){
    	this.sql  = sql;
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.COUNT;
    	return this;
    }
    
    /**
     * @return	返回更新model对象，推荐方式
     */
    public Model update(){
    	this.sql = "update " + getTable();
    	this.params = CollectionKit.newHashMap();
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.UPDATE;
    	return this;
    }
    
    /**
     * 返回更新model对象
     * 
     * @param sql	自定义更新语句
     * @return		返回更新model对象
     */
    public Model update(String sql){
    	this.sql = sql;
    	this.params = CollectionKit.newHashMap();
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.UPDATE;
    	return this;
    }
    
    /**
     * @return	返回插入model对象，推荐方式
     */
    public Model insert(){
    	this.sql = "insert into " + getTable();
    	this.params = CollectionKit.newHashMap();
    	this.dmlType = DmlType.INSERT;
    	return this;
    }
    
    /**
     * 根据自定义sql返回插入model对象
     * @param sql	自定义插入语句
     * @return		返回插入model对象
     */
    public Model insert(String sql){
    	this.sql = sql;
    	this.params = CollectionKit.newHashMap();
    	this.dmlType = DmlType.INSERT;
    	return this;
    }
    
    /**
     * @return	返回删除model对象
     */
    public Model delete(){
    	this.sql = "delete from " + getTable();
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.DELETE;
    	return this;
    }
    
    /**
     * 返回自定义删除model对象
     * 
     * @param sql	自定义删除语句
     * @return		返回自定义删除model对象
     */
    public Model delete(String sql){
    	this.sql = sql;
    	this.whereParams = CollectionKit.newHashMap();
    	this.dmlType = DmlType.DELETE;
    	return this;
    }
    
    /**
     * 设置参数列表，新增，更新用到
     * 
     * @param name	参数键
     * @param value	参数值
     * @return		返回model对象
     */
    public Model param(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		this.params.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 设置where参数列表，查询，更新，删除用到
     * 
     * @param name	参数键
     * @param value	参数值
     * @return		返回model对象
     */
    public Model where(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		this.whereParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 大于
     * @param name
     * @param value
     * @return
     */
    public Model greater(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		
    		if(null == greaterParams){
    			greaterParams = CollectionKit.newHashMap();
        	}
    		
    		this.greaterParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 大于等于
     * 
     * @param name	参数键
     * @param value	参数值
     * @return		返回model对象
     */
    public Model greaterThan(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		
    		if(null == greaterThanParams){
    			greaterThanParams = CollectionKit.newHashMap();
        	}
    		
    		this.greaterThanParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 小于
     * @param name	参数键
     * @param value	参数值
     * @return		返回model对象
     */
    public Model less(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		
    		if(null == lessParams){
    			lessParams = CollectionKit.newHashMap();
        	}
    		
    		this.lessParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 小于等于
     * @param name	参数键
     * @param value	参数值
     * @return		返回model对象
     */
    public Model lessThan(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		
    		if(null == lessThanParams){
    			lessThanParams = CollectionKit.newHashMap();
        	}
    		
    		this.lessThanParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * like
     * @param name
     * @param value
     * @return
     */
    public Model like(String name, Object value){
    	if(StringKit.isNotBlank(name) && null != value){
    		if(null == likeParams){
    			likeParams = CollectionKit.newHashMap();
        	}
    		this.likeParams.put(name, value);
    	}
    	return this;
    }
    
    /**
     * 设置排序规则
     * 
     * @param orderby	排序字段和排序规则，如：ordernum desc
     * @return			返回model对象
     */
    public Model orderBy(String orderby){
    	this.orderby = orderby;
    	return this;
    }
    
    private String getConditionSql(String sql){
    	
    	if(null == sql){
    		sql = this.sql;
    	}
    	
    	StringBuffer sqlBuf = new StringBuffer(sql);
		String sqlEnd = sql;
		
		// 基础where equals条件
		if(null != whereParams && whereParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = whereParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + "=:" + name + " and ");
			}
		}
		
		// 大于条件
		if(null != greaterParams && greaterParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = greaterParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + ">:" + name + " and ");
			}
		}
		
		// 小于条件
		if(null != lessParams && lessParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = lessParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + "<:" + name + " and ");
			}
		}
		
		// 大于等于条件
		if(null != greaterThanParams && greaterThanParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = greaterThanParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + ">=:" + name + " and ");
			}
		}
		
		// 小于等于条件
		if(null != lessThanParams && lessThanParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = lessThanParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + "<=:" + name + " and ");
			}
		}
		
		// like条件
		if(null != likeParams && likeParams.size() > 0){
			if(sqlBuf.indexOf("where") == -1){
				sqlBuf.append(" where ");
			}
			Set<String> keys = likeParams.keySet();
			for(String name : keys){
				sqlBuf.append(name + "like:" + name + " and ");
			}
		}
		
		if(sqlBuf.lastIndexOf(" and ") != -1){
			sqlEnd = sqlBuf.substring(0, sqlBuf.length() - 5);
		}
		return sqlEnd;
    }
    
    private Query parseParams(Query query){
    	
    	// insert、update参数
    	if(null != params && params.size() > 0){
			Set<String> keys = params.keySet();
			for(String name : keys){
				query.addParameter(name, params.get(name));
			}
		}
    	
    	// 基础where equals条件
		if(null != whereParams && whereParams.size() > 0){
			Set<String> keys = whereParams.keySet();
			for(String name : keys){
				query.addParameter(name, whereParams.get(name));
			}
		}
		
		// 大于条件
		if(null != greaterParams && greaterParams.size() > 0){
			Set<String> keys = greaterParams.keySet();
			for(String name : keys){
				query.addParameter(name, greaterParams.get(name));
			}
		}
		
		// 小于条件
		if(null != lessParams && lessParams.size() > 0){
			Set<String> keys = lessParams.keySet();
			for(String name : keys){
				query.addParameter(name, lessParams.get(name));
			}
		}
		
		// 大于等于条件
		if(null != greaterThanParams && greaterThanParams.size() > 0){
			Set<String> keys = greaterThanParams.keySet();
			for(String name : keys){
				query.addParameter(name, greaterThanParams.get(name));
			}
		}
		
		// 小于等于条件
		if(null != lessThanParams && lessThanParams.size() > 0){
			Set<String> keys = lessThanParams.keySet();
			for(String name : keys){
				query.addParameter(name, lessThanParams.get(name));
			}
		}
		
		// like条件
		if(null != likeParams && likeParams.size() > 0){
			Set<String> keys = likeParams.keySet();
			for(String name : keys){
				query.addParameter(name, likeParams.get(name));
			}
		}
		
		return query;
    }
    
    /**
     * @return	返回记录数
     */
    public Long fetchCount(){
    	if(dmlType.equals(DmlType.COUNT) || dmlType.equals(DmlType.SELECT)){
    		
    		String sqlEnd = getConditionSql(this.sql);
    		
    		if(sqlEnd.indexOf("* from") != -1){
    			sqlEnd = sqlEnd.replaceFirst("\\*", "count(1)");
    		}
    		
    		Connection conn = sql2o.open();
    		Query query = conn.createQuery(sqlEnd);
    		
    		query = parseParams(query);
    		
    		LOGGER.debug("execute sql：" + query.toString());
    		LOGGER.debug("execute parameter：" + whereParams.values());
    		return query.executeScalar(Long.class);
    	}
    	
    	return 0L;
    }
    
    /**
     * @return		返回查询一个对象
     */
    @SuppressWarnings("unchecked")
	public <M extends Model> M fetchOne(){
    	if(this.dmlType.equals(DmlType.SELECT)){
    		
    		String sqlEnd = getConditionSql(this.sql);
    		
    		Connection conn = sql2o.open();
    		Query query = conn.createQuery(sqlEnd);
    		query = parseParams(query);
    		
    		LOGGER.debug("execute sql：" + query.toString());
    		LOGGER.debug("execute parameter：" + whereParams.values());
    		return (M) query.executeAndFetchFirst(model);
    	}
    	return null;
    }
    
    /**
     * @return		返回查询一个对象
     */
    @SuppressWarnings("unchecked")
	public <M> M fetchColum(){
    	if(this.dmlType.equals(DmlType.SELECT)){
    		
    		String sqlEnd = getConditionSql(this.sql);
    		
    		Connection conn = sql2o.open();
    		Query query = conn.createQuery(sqlEnd);
    		query = parseParams(query);
    		
    		LOGGER.debug("execute sql：" + query.toString());
    		LOGGER.debug("execute parameter：" + whereParams.values());
    		return (M) query.executeScalar();
    	}
    	return null;
    }
    
    /**
     * @return	返回查询的list
     */
    @SuppressWarnings("unchecked")
	public <M extends Model> List<M> fetchList(){
    	if(this.dmlType.equals(DmlType.SELECT)){
    		
    		String sqlEnd = getConditionSql(sql);
    		
    		if(null != this.orderby){
    			sqlEnd += " order by " + orderby;
    		}
    		Connection conn = sql2o.open();
    		Query query = conn.createQuery(sqlEnd);
    		query = parseParams(query);
    		
    		LOGGER.debug("execute sql：" + query.toString());
    		LOGGER.debug("execute parameter：" + whereParams.values());
    		return (List<M>) query.executeAndFetch(model);
    	}
    	return null;
    }
    
    /**
     * 分页检索
     * 
     * @param page		当前页码
     * @param pageSize	每页条数
     * @return			返回分页后的Page<M>对象
     */
    @SuppressWarnings("unchecked")
	public <M extends Model> Page<M> fetchPage(Integer page, Integer pageSize){
    	
    	Page<M> pageModel = new Page<M>(0, page, pageSize);
    	
    	if(this.dmlType.equals(DmlType.SELECT) && null != page && null != pageSize && page > 0 && pageSize > 0){
    		
    		// 查询总记录数
    		long totalCount = fetchCount();
    		
    		String sqlEnd = getConditionSql(this.sql);
    				
    		if(null != this.orderby){
    			sqlEnd += " order by " + orderby;
    		}
    		
    		sqlEnd += " limit :page, :pageSize";
    		
    		this.whereParams.put("page", page - 1);
			this.whereParams.put("pageSize", pageSize);
			
			// 设置query
			Connection conn = sql2o.open();
    		Query query = conn.createQuery(sqlEnd);
    		query = parseParams(query);
    		
    		pageModel = new Page<M>(totalCount, page, pageSize);
    		
    		LOGGER.debug("execute sql：" + query.toString());
    		LOGGER.debug("execute parameter：" + whereParams.values());
    		
    		List<? extends Model> results = query.executeAndFetch(this.model);
			if(null != results && results.size() > 0){
				pageModel.setResults((List<M>) results);
			}
			
    		return pageModel;
    	}
    	
    	return pageModel;
    }
    
    /**
     * 执行并提交，事务一致
     */
    public void executeAndCommit(){
    	
    	// 插入
    	if(this.dmlType.equals(DmlType.INSERT)){
    		insertCommit(null);
    	}
    	
    	// 更新
    	if(this.dmlType.equals(DmlType.UPDATE)){
    		updateCommit(null);
    	}
    	
    	// 删除
    	if(this.dmlType.equals(DmlType.DELETE)){
    		deleteCommit(null);
    	}
    	
    }
    
    /**
     * 执行并提交，事务一致
     */
    public void execute(Connection connection){
    	
    	if(null == connection){
    		connection = sql2o.beginTransaction();
    	}
    	
    	// 插入
    	if(this.dmlType.equals(DmlType.INSERT)){
    		insertCommit(connection);
    	}
    	
    	// 更新
    	if(this.dmlType.equals(DmlType.UPDATE)){
    		updateCommit(connection);
    	}
    	
    	// 删除
    	if(this.dmlType.equals(DmlType.DELETE)){
    		deleteCommit(connection);
    	}
    	
    }
    
    /**
     * @return	删除并返回连接
     */
    private Connection deleteCommit(Connection conn) {
    	
		String deleteSql = getConditionSql(this.sql);
		
		if(null == conn){
			conn = sql2o.open();
		}
		
		Query query = conn.createQuery(deleteSql);
		query = parseParams(query);
		
		LOGGER.debug("execute sql：" + query.toString());
		LOGGER.debug("execute parameter：" + whereParams.values());
		
        return query.executeUpdate();
    	
	}

    /**
     * @return	插入并返回连接
     */
	private Connection insertCommit(Connection conn){
		
    	String insertSql = this.sql;
    	
		if(!params.isEmpty() && params.size() > 0){
			
			StringBuffer paramBuf = new StringBuffer("(");
			StringBuffer valuesBuf = new StringBuffer(" values(");
			
			Set<String> keys = params.keySet();
			for(String name : keys){
				paramBuf.append(name + ", ");
				valuesBuf.append(":" + name + ", ");
			}
			
			if(paramBuf.lastIndexOf(", ") != -1 && valuesBuf.lastIndexOf(", ") != -1){
				String start = paramBuf.substring(0, paramBuf.length() - 2);
				String end = valuesBuf.substring(0, valuesBuf.length() - 2);
				
				insertSql = this.sql + start + ") " + end + ")";
			}
		}
		
		if(null == conn){
			conn = sql2o.open();
		}
		
		Query query = conn.createQuery(insertSql);
		query = parseParams(query);
		
		LOGGER.debug("execute sql：" + query.toString());
		LOGGER.debug("execute parameter：" + params.values());
		
		return query.executeUpdate();
    }
    
	/**
     * @return	更新并返回连接
     */
    private Connection updateCommit(Connection conn){
    	
    	if(null != params && params.size() > 0){
			
			StringBuffer setBuf = new StringBuffer(" set ");
			StringBuffer whereBuf = new StringBuffer("(");
			
			String setSql = "";
			String whereSql = "";
			
			Set<String> keys = params.keySet();
			for(String name : keys){
				setBuf.append(name + "=:" + name);
			}
			
			if(setBuf.lastIndexOf(", ") != -1){
				setSql = setBuf.substring(0, setBuf.length() - 2);
			}
			
			whereSql = getConditionSql(this.sql);
			
			/*if(whereParams.size() > 0){
				Set<String> wkeys = whereParams.keySet();
    			for(String name : wkeys){
    				whereBuf.append(name + " = :" + name + ", ");
    			}
			}*/
			
			if(whereBuf.lastIndexOf(", ") != -1){
				whereSql = whereBuf.substring(0, whereBuf.length() - 2);
			}
			
			if(setSql.length() > 0){
				
				String updateSql = this.sql + setSql + whereSql;
				
				if(null == conn){
					conn = sql2o.open();
				}
				
	    		Query query = conn.createQuery(updateSql);
	    		query = parseParams(query);
				
				LOGGER.debug("execute sql：" + query.toString());
	    		LOGGER.debug("execute parameter：" + params.values() + whereParams.values());
	    		
				return query.executeUpdate();
			}
		}
		return null;
    }
    
    public List<Map<String, Object>> tableAsList(org.sql2o.data.Table table) {
		if(null != table){
			List<Map<String, Object>> list = table.asList();
			
			List<Map<String, Object>> result = new ArrayList<Map<String,Object>>(list.size());
			if(null != list && list.size() > 0){
				for(Map<String, Object> map : list){
					Map<String, Object> m = getMap(map);
					if(null != m && m.size() > 0){
						result.add(m);
					}
				}
				
				return result;
			}
		}
		return null;
	}
    
    private Map<String, Object> getMap(Map<String, Object> m) {
		// 一个map就是数据库一行记录
		Set<String> names = m.keySet();
		if(null != names && names.size() > 0){
			Map<String, Object> map = new HashMap<String, Object>(names.size());
			
			for(String name : names){
				map.put(name, m.get(name));
			}
			
			return map;
		}
		return null;
	}
}
