/**
 * Copyright (c) 2005-2012 springside.org.cn
 */
package com.ckjava.utils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;

/**
 * 封装各种格式的编码解码工具类.
 * 1.Commons-Codec的 hex/base64 编码
 * 2.自制的base62 编码
 * 3.Commons-Lang的xml/html escape
 * 4.JDK提供的URLEncoder

 */
public class EncodesUtils {

	public static final String DEFAULT_ENCODE = "UTF-8";
	private static final char[] BASE62 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".toCharArray();

	/**
	 *  将 {@code byte[]} 转成 Hex 编码
	 * 
	 * @param input {@code byte[]}
	 * @return Hex 编码
	 */
	public static String encodeHex(byte[] input) {
		return new String(Hex.encodeHex(input));
	}

	/**
	 * Hex解码.
	 * 
	 * @param input String 类型
	 * @return byte[] 类型
	 * @throws DecoderException 解码异常
	 */
	public static byte[] decodeHex(String input) throws DecoderException {
		return Hex.decodeHex(input.toCharArray());
	}

	/**
	 * Base64编码.
	 * 
	 * @param input byte[] 类型
	 * @return String 类型
	 */
	public static String encodeBase64(byte[] input) {
		return new String(Base64.encodeBase64(input));
	}
	
	/**
	 * Base64编码
	 * 
	 * @param input String 类型, 以 UTF-8 编码
	 * @return String 类型
	 * @throws UnsupportedEncodingException 未知的字符编码异常
	 */
	public static String encodeBase64(String input) throws UnsupportedEncodingException {
		return new String(Base64.encodeBase64(input.getBytes(DEFAULT_ENCODE)));
	}

	/**
	 * Base64解码.
	 * 
	 * @param input String 类型
	 * @return byte[] 类型
	 * @throws UnsupportedEncodingException 未知编码异常
	 */
	public static byte[] decodeBase64(String input) throws UnsupportedEncodingException {
		return Base64.decodeBase64(input.getBytes(DEFAULT_ENCODE));
	}
	
	/**
	 * Base64解码.
	 * 
	 * @param input String 类型
	 * @return String 类型
	 * @throws UnsupportedEncodingException 未知编码异常
	 */
	public static String decodeBase64String(String input) throws UnsupportedEncodingException {
		return new String(Base64.decodeBase64(input.getBytes(DEFAULT_ENCODE)), DEFAULT_ENCODE);
	}

	/**
	 * Base62编码
	 * 
	 * @param input byte[]类型
	 * @return String 类型
	 */
	public static String encodeBase62(byte[] input) {
		char[] chars = new char[input.length];
		for (int i = 0; i < input.length; i++) {
			chars[i] = BASE62[((input[i] & 0xFF) % BASE62.length)];
		}
		return new String(chars);
	}

	/**
	 * URL 编码, Encode默认为UTF-8. 
	 * 
	 * @param part String 类型
	 * @return String 类型 
	 * @throws UnsupportedEncodingException 未知编码异常
	 */
	public static String urlEncode(String part) throws UnsupportedEncodingException {
		return URLEncoder.encode(part, DEFAULT_ENCODE);
	}

	/**
	 * URL 解码, Encode默认为UTF-8. 
	 * 
	 * @param part String 类型
	 * @return String 类型
	 * @throws UnsupportedEncodingException 未知编码异常
	 */
	public static String urlDecode(String part) throws UnsupportedEncodingException {
		return URLDecoder.decode(part, DEFAULT_ENCODE);
	}
}
