/*
 * Copyright 2010-2013, CloudBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.util;

import com.google.common.base.Preconditions;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;
import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

/**
 * XML Utils.
 *
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class XmlUtils {

    private final static XPath xpath = XPathFactory.newInstance().newXPath();
    private final static DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
    private final static TransformerFactory transformerFactory = TransformerFactory.newInstance();

    static {
        documentBuilderFactory.setValidating(false);
        try {
            documentBuilderFactory.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Exception disabling 'http://apache.org/xml/features/nonvalidating/load-external-dtd'", e);
        }
    }

    /**
     * Returns the unique XML element matching the given {@code xpathExpression}
     *
     * @param document
     * @param xpathExpression xpath expression to find the element
     * @return the matching element
     * @throws RuntimeException 0 or more than 1 matching element found
     */
    @Nonnull
    public static Element getUniqueElement(@Nonnull Document document, @Nonnull String xpathExpression) throws RuntimeException {
        return getUniqueElement((Node) document, xpathExpression);

    }

    /**
     * Returns the unique XML element matching the given {@code xpathExpression}
     *
     * @param element
     * @param xpathExpression xpath expression to find the element
     * @return the matching element
     * @throws RuntimeException 0 or more than 1 matching element found
     */
    @Nonnull
    public static Element getUniqueElement(@Nonnull Element element, @Nonnull String xpathExpression) {
        return getUniqueElement((Node) element, xpathExpression);
    }

    /**
     * Returns the unique XML element matching the given {@code xpathExpression}
     *
     * @param node
     * @param xpathExpression xpath expression to find the node
     * @return the matching node
     * @throws RuntimeException 0 or more than 1 matching element found
     */
    @Nonnull
    public static Element getUniqueElement(@Nonnull Node node, @Nonnull String xpathExpression) {
        try {
            NodeList nl = (NodeList) xpath.compile(xpathExpression).evaluate(node, XPathConstants.NODESET);
            if (nl.getLength() == 0 || nl.getLength() > 1) {
                throw new RuntimeException("More or less (" + nl.getLength() + ") than 1 node found for expression: " + xpathExpression);
            }
            return (Element) nl.item(0);
        } catch (Exception e) {
            throw new RuntimeException("Exception evaluating xpath '" + xpathExpression + "' on " + node, e);
        }
    }

    /**
     * Load the given {@code file} as an XML document.
     *
     * @param file
     * @return the loaded XML document
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    @Nonnull
    public static Document loadXmlDocumentFromFile(@Nonnull File file) throws ParserConfigurationException, IOException, SAXException {
        Preconditions.checkArgument(file.exists(), "File not found %s", file);
        Preconditions.checkArgument(!file.isDirectory(), "Expected file and not directory: %s", file);

        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(file);
    }

    /**
     * Load the given {@code path} as an XML document.
     *
     * @param path
     * @return the loaded XML document
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    @Nonnull
    public static Document loadXmlDocumentFromPath(@Nonnull Path path) throws ParserConfigurationException, IOException, SAXException {
        Preconditions.checkArgument(Files.exists(path), "File not found %s", path);
        Preconditions.checkArgument(!Files.isDirectory(path), "Expected file and not directory: %s", path);

        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(Files.newInputStream(path));
    }

    /**
     * Load the given {@link InputStream} as an XML document.
     *
     * @param in
     * @return the loaded XML document
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    @Nonnull
    public static Document loadXmlDocumentFromStream(@Nonnull InputStream in) throws ParserConfigurationException, IOException, SAXException {
        Preconditions.checkNotNull(in, "Inputstream can not be null");
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(in);
    }

    /**
     * Check that the name of root element of the given {@code document} matches the given {@code expectedRootElementName}
     *
     * @param document
     * @param expectedRootElementName
     */
    public static void checkRootElement(@Nonnull Document document, @Nullable String expectedRootElementName) {
        if (document.getDocumentElement() == null || expectedRootElementName == null) {
            return;
        } else if (!expectedRootElementName.equals(document.getDocumentElement().getNodeName())) {
            throw new IllegalStateException("Invalid root element '" + document.getDocumentElement().getNodeName() + "', expected '" + expectedRootElementName + "'");

        }
    }

    /**
     * Serialise the given {@link Document} into a {@code String}.
     *
     * @param in
     * @return serialized XML
     * @throws TransformerRuntimeException
     */
    @Nonnull
    public static String flush(@Nonnull Document in) throws TransformerRuntimeException {
        return flush((Node) in);
    }

    /**
     * Serialise the given {@link Document} into a {@code String}.
     *
     * @param in
     * @return serialized XML
     * @throws TransformerRuntimeException
     */
    @Nonnull
    public static String flush(@Nonnull Node in) throws TransformerRuntimeException {
        StringWriter sw = new StringWriter();
        flush(in, sw);
        return sw.toString();
    }

    /**
     * Serialise the given {@link Document} in the given {@link OutputStream}.
     *
     * @param in
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Document in, @Nonnull OutputStream out) throws TransformerRuntimeException {
        flush((Node) in, out);
    }

    /**
     * Serialise the given {@link Node} in the given {@link OutputStream}.
     *
     * @param in
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Node in, @Nonnull OutputStream out) throws TransformerRuntimeException {
        flush(in, out, new HashMap<String, String>());
    }

    /**
     * Serialise the given {@link Document} in the given {@link Writer}.
     *
     * @param in
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Document in, @Nonnull Writer out) throws TransformerRuntimeException {
        flush((Node) in, out);
    }

    /**
     * Serialise the given {@link Node} in the given {@link Writer}.
     *
     * @param in
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Node in, @Nonnull Writer out) throws TransformerRuntimeException {
        flush(in, out, new HashMap<String, String>());
    }

    /**
     * Serialise the given {@link Document} in the given {@link OutputStream}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Document in, @Nonnull OutputStream out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        flush((Node) in, out, outputProperties);
    }

    /**
     * Serialise the given {@link Node} in the given {@link OutputStream}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Node in, @Nonnull OutputStream out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        flush(in, new StreamResult(out), outputProperties);
        try {
            out.flush();
        } catch (IOException e) {
            throw new RuntimeException("Exception flushing document", e);
        }
    }

    /**
     * Serialise the given {@link Document} in the given {@link Writer}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Document in, @Nonnull Writer out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        flush((Node) in, out, outputProperties);
    }

    /**
     * Serialise the given {@link Node} in the given {@link Writer}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Node in, @Nonnull Writer out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        flush(in, new StreamResult(out), outputProperties);
        try {
            out.flush();
        } catch (IOException e) {
            throw new RuntimeException("Exception flushing document", e);
        }
    }

    /**
     * Serialise the given {@link Document} in the given {@link Result}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Document in, @Nonnull Result out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        flush((Node) in, out, outputProperties);
    }

    /**
     * Serialise the given {@link Node} in the given {@link Result}.
     *
     * @param in
     * @param outputProperties see {@link Transformer#setOutputProperty(String, String)}
     * @param out
     * @throws TransformerRuntimeException
     */
    public static void flush(@Nonnull Node in, @Nonnull Result out, @Nullable Map<String, String> outputProperties) throws TransformerRuntimeException {
        try {
            // Write the content into XML file
            Transformer transformer = transformerFactory.newTransformer();

            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
            transformer.setOutputProperty(OutputKeys.STANDALONE, "no");

            if (outputProperties != null) {
                for (Map.Entry<String, String> property : outputProperties.entrySet()) {
                    transformer.setOutputProperty(property.getKey(), property.getValue());
                }
            }

            transformer.transform(new DOMSource(in), out);
        } catch (TransformerException e) {
            throw new TransformerRuntimeException("Exception flushing document", e);
        }
    }

    /**
     * Insert the given {@code newElement} just next to the given {@code sibling}, after it.
     *
     * @param newElement
     * @param sibling
     */
    public static void insertSiblingAfter(@Nonnull Element newElement, @Nonnull Element sibling) {
        Node nextSibling = sibling.getNextSibling();
        sibling.getParentNode().insertBefore(newElement, nextSibling);
    }

    /**
     * @param element root element to locate the element to modify
     * @param xpath   xpath expression to locate the element to modify
     * @param value   no value of the element
     */
    public static void setElementTextContent(@Nonnull Element element, @Nonnull String xpath, @Nullable String value) {
        Element elementToUpdate = XmlUtils.getUniqueElement(element, xpath);
        elementToUpdate.setTextContent(value);
    }

    /**
     * Runtime version of {@link TransformerException}
     */
    public static class TransformerRuntimeException extends RuntimeException {
        public TransformerRuntimeException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}
