/*
 * Copyright 2010-2013, the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.domain.environment;

import javax.annotation.Nonnull;
import java.nio.file.Path;

/**
 * <p>See <a href="http://genapp-docs.cloudbees.com/plugins.html#setup-environment-variables">
 * genapp documentation / plugins / Setup Environment Variables</a></p>
 *
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class Environment {
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-dir">app_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path appDir;
    /**
     * Aka {@code plugin_dir}
     */
    @Nonnull
    public final Path clickstackDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#pkg-dir">pkg_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path packageDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-user">app_user</a>:
     * The user associated with the application being deployed</p>
     */
    @Nonnull
    public final String appUser;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-id">app_id</a>:
     * The unique ID of the application being deployed</p>
     */
    @Nonnull
    public final String appId;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-port">app_port</a>:
     * The genapp assigned port for the application
     * <p/>
     */
    @Nonnull
    public final int appPort;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#control-dir">control_dir</a>:
     * The control subdirectory located within the genapp directory</p>
     */
    @Nonnull
    public final Path controlDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#log-dir">log_dir</a>:
     * The conventional location for application logs</p>
     */
    @Nonnull
    public final Path logDir;
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#genapp-dir"">genapp_dir</a>:
     * The genapp subdirectory located within the application directory
     * </p>
     * <p>This variable is equivalent to {@code $app_dir/.genapp}.</p>
     * <p>Plugins may use this variable to reference files located in the genapp subdirectory.
     * These may be files created by other plugins that have already performed their setup operations or files that the plugin itself creates.
     * </p>
     */
    @Nonnull
    public final Path genappDir;

    public Environment(
            @Nonnull Path appDir,
            @Nonnull Path clickstackDir,
            @Nonnull Path packageDir,
            @Nonnull String appUser,
            @Nonnull String appId,
            @Nonnull int appPort,
            @Nonnull Path controlDir,
            @Nonnull Path logDir) {
        this.appDir = appDir;
        this.clickstackDir = clickstackDir;
        this.packageDir = packageDir;
        this.appUser = appUser;
        this.appId = appId;
        this.appPort = appPort;
        this.controlDir = controlDir;
        this.logDir = logDir;
        this.genappDir = appDir.resolve(".genapp");


    }

    @Override
    public String toString() {
        return "Environment{" + ", \n" +
                " appUser='" + appUser + '\'' +", \n" +
                " appId='" + appId + '\'' +", \n" +
                " appPort=" + appPort +", \n" +
                " appDir=" + appDir.toAbsolutePath() +", \n" +
                " logDir=" + logDir.toAbsolutePath() +", \n" +
                " genappDir=" + genappDir.toAbsolutePath() +", \n" +
                " controlDir=" + controlDir.toAbsolutePath() +", \n" +
                " clickstackDir=" + clickstackDir.toAbsolutePath() +", \n" +
                " packageDir=" + packageDir.toAbsolutePath() +", \n" +
                '}';
    }
}
