/*
 * Copyright 2010-2013, the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.domain.environment;

import com.cloudbees.clickstack.domain.metadata.Metadata;
import com.cloudbees.clickstack.util.Files2;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>See <a href="http://genapp-docs.cloudbees.com/plugins.html#setup-environment-variables">
 * genapp documentation / plugins / Setup Environment Variables</a></p>
 *
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class Environment {
    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-dir">app_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path appDir;

    /**
     * directory that contains the plugin's setup script, aka {@code plugin_dir}
     */
    @Nonnull
    public final Path clickstackDir;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#pkg-dir">pkd_dir</a>:
     * The application source package directory</p>
     */
    @Nonnull
    public final Path packageDir;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-user">app_user</a>:
     * The user associated with the application being deployed</p>
     */
    @Nonnull
    public final String appUser;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-id">app_id</a>:
     * The unique ID of the application being deployed</p>
     */
    @Nonnull
    public final String appId;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#app-port">app_port</a>:
     * The genapp assigned port for the application
     * <p/>
     */
    public final int appPort;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#control-dir">control_dir</a>:
     * The control subdirectory located within the genapp directory</p>
     */
    @Nonnull
    public final Path controlDir;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#log-dir">log_dir</a>:
     * The conventional location for application logs</p>
     */
    @Nonnull
    public final Path logDir;

    /**
     * <p><a href="http://genapp-docs.cloudbees.com/plugins.html#genapp-dir"">genapp_dir</a>:
     * The genapp subdirectory located within the application directory
     * </p>
     * <p>This variable is equivalent to {@code $appDir/.genapp}.</p>
     * <p>Plugins may use this variable to reference files located in the genapp subdirectory.
     * These may be files created by other plugins that have already performed their setup operations or files that the plugin itself creates.
     * </p>
     */
    @Nonnull
    public final Path genappDir;

    /**
     * Convenient directory for application to embed custom extra files, like runtime dependencies.
     * <p>
     * This variable is equivalent to {@code $appDir/app-extra-files}
     */
    @Nonnull
    private final Path appExtraFiles;

    public Environment(@Nonnull Path appDir,
                       @Nonnull Path clickstackDir,
                       @Nonnull Path packageDir,
                       @Nonnull String appUser,
                       @Nonnull String appId,
                       int appPort) {
        this(appDir, clickstackDir, packageDir,
                appUser, appId, appPort, appDir.resolve(".genapp/control"), appDir.resolve(".genapp/log"));
    }

    public Environment(
            @Nonnull Path appDir,
            @Nonnull Path clickstackDir,
            @Nonnull Path packageDir,
            @Nonnull String appUser,
            @Nonnull String appId,
            int appPort,
            @Nonnull Path controlDir,
            @Nonnull Path logDir) {
        this.appDir = appDir;
        this.clickstackDir = clickstackDir;
        this.packageDir = packageDir;
        this.appUser = appUser;
        this.appId = appId;
        this.appPort = appPort;
        this.controlDir = controlDir;
        this.logDir = logDir;
        this.genappDir = appDir.resolve(".genapp");
        this.appExtraFiles = appDir.resolve("app-extra-files");
    }

    @Override
    public String toString() {
        return "Environment{" + ", \n" +
                " appUser='" + appUser + '\'' +", \n" +
                " appId='" + appId + '\'' +", \n" +
                " appPort=" + appPort +", \n" +
                " appDir=" + appDir.toAbsolutePath() +", \n" +
                " logDir=" + logDir.toAbsolutePath() +", \n" +
                " genappDir=" + genappDir.toAbsolutePath() +", \n" +
                " controlDir=" + controlDir.toAbsolutePath() +", \n" +
                " clickstackDir=" + clickstackDir.toAbsolutePath() +", \n" +
                " packageDir=" + packageDir.toAbsolutePath() +", \n" +
                '}';
    }

    public static final String DEFAULT_JAVA_VERSION = "1.7";

    @Nonnull
    public Path getJavaExecutable(Metadata metadata) throws IOException {
        Path javaPath = getJavaHome(metadata).resolve("bin/java");
        Preconditions.checkState(Files.exists(javaPath), "Java executable %s does not exist");
        Preconditions.checkState(!Files.isDirectory(javaPath), "Java executable %s is not a file");
        return javaPath;
    }

    @Nonnull
    public Path getJavaHome(Metadata metadata) throws IOException {
        String javaVersion = metadata.getRuntimeParameter("java", "version", DEFAULT_JAVA_VERSION);
        Path javaHome;
        switch (javaVersion) {
            case "6":
            case "1.6": javaHome = FileSystems.getDefault().getPath("/opt/java6"); break;
            case "7":
            case "1.7": javaHome = FileSystems.getDefault().getPath("/opt/java7"); break;
            case "8":
            case "1.8": javaHome = FileSystems.getDefault().getPath("/opt/java8"); break;
            case "custom":

                LOGGER.warn("**WARNING** you're running with a custom JRE, at your own risk");
                String path = expand(metadata.getRuntimeParameter("java", "home"));
                javaHome = FileSystems.getDefault().getPath(path);

                Files2.chmodAddReadExecute(javaHome.resolve("bin"));
                break;
            default:
                throw new RuntimeException("Unsupported java version "+javaVersion);
        }

        Preconditions.checkState(Files.exists(javaHome), "JavaHome %s does not exist");
        Preconditions.checkState(Files.isDirectory(javaHome), "JavaHome %s is not a directory");
        return javaHome;
    }


    private static final Pattern VARIABLE = Pattern.compile("\\$([A-Za-z0-9_]+|\\{[A-Za-z0-9_.]+\\}|\\$)");

    /**
     * Expand variables in string as <tt>$variable</tt> or <tt>${variable}</tt>/ Invalid variables are left as-is.
     * <p>
     * Adapted from hudson.Util#replaceMacro
     */
    public String expand(@Nullable String s) {
        if (s == null) {
            return null;
        }

        int idx=0;
        while(true) {
            Matcher m = VARIABLE.matcher(s);
            if(!m.find(idx))   return s;String key = m.group().substring(1);

            // escape the dollar sign or get the key to resolve
            String value;
            if(key.charAt(0)=='$') {
                value = "$";
            } else {
                if(key.charAt(0)=='{')  key = key.substring(1,key.length()-1);
                value = resolve(key);
            }

            if(value==null)
                idx = m.end(); // skip this
            else {
                s = s.substring(0,m.start())+value+s.substring(m.end());
                idx = m.start() + value.length();
            }
        }
    }

    private String resolve(String key) {
        switch (key) {
            case "app_dir": return appDir.toAbsolutePath().toString();
            case "plugin_dir": return clickstackDir.toAbsolutePath().toString();
            case "pkg_dir": return packageDir.toAbsolutePath().toString();
            case "app_user": return appUser;
            case "app_id": return appId;
            case "app_port": return String.valueOf(appPort);
            case "control_dir": return controlDir.toAbsolutePath().toString();
            case "log_dir": return logDir.toAbsolutePath().toString();
            case "genapp_dir": return genappDir.toAbsolutePath().toString();
            case "app-extra-files": return appExtraFiles.toAbsolutePath().toString();
            default: return key;
        }
    }

    private final static Logger LOGGER = LoggerFactory.getLogger(Environment.class);
}
