/*
 * Copyright 2010-2013, CloudBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cloudbees.clickstack.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.net.URL;
import java.security.CodeSource;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

/**
 * @author <a href="mailto:cleclerc@cloudbees.com">Cyrille Le Clerc</a>
 */
public class Manifests {

    private final static Logger logger = LoggerFactory.getLogger(Manifests.class);

    /**
     * <p>
     * Returns the manifest of the jar file that contains the given {@code clazz}.
     * </p>
     * <p>If the MANIFEST is not found, {@code null} is returned.</p>
     *
     * @param clazz
     * @return the MANIFEST associated with the given {@code clazz}
     */
    @Nullable
    public static Manifest getManifest(@Nonnull Class clazz) {
        CodeSource codeSource = clazz.getProtectionDomain().getCodeSource();
        if (codeSource == null) {
            logger.trace("No CodeSource found for {}", clazz);
            return null;
        }

        URL location = codeSource.getLocation();
        File file = new File(location.getFile());
        if (!file.exists()) {
            logger.warn("CodeSource location for {} does not exist: {}", clazz, location.getFile());
            return null;
        }

        try {
            logger.trace("Load jar {}", file);
            JarFile jarFile = new JarFile(file);
            return jarFile.getManifest();
        } catch (Exception e) {
            logger.warn("Exception retrieving manifest for {} in {}", clazz, location.getFile(), e);
            return null;
        }
    }

    @Nullable
    public static String getAttribute(@Nonnull Class clazz, @Nonnull String attribute, @Nullable String defaultValue) {
        Manifest manifest = getManifest(clazz);
        if (manifest == null) {
            logger.debug("No manifest found for {}", clazz);
            return defaultValue;
        }

        String value = manifest.getMainAttributes().getValue(attribute);
        if (value == null) {
            value = defaultValue;
        }
        return value;
    }

    @Nullable
    public static String getAttribute(@Nonnull Class clazz, @Nonnull String attribute) {
        return getAttribute(clazz, attribute, null);
    }
}
