package com.codeupsoft.base.common.enums;

import com.baomidou.mybatisplus.annotation.IEnum;
import com.codeupsoft.base.common.domain.Options;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 基础枚举接口，提供通用的枚举处理方法.
 *
 * @param <T> 枚举值类型，需实现Serializable接口
 * @author Liu,Dongdong
 * @since 1.0.0
 */
@SuppressWarnings("unused")
public interface BaseEnum<T extends Serializable> extends IEnum<T> {

  /**
   * 根据枚举值获取对应的枚举实例.
   *
   * @param value 枚举值
   * @param clazz 枚举类型Class对象
   * @param <E> 枚举类型
   * @param <T> 枚举值类型
   * @return 匹配的枚举实例，未找到返回null
   */
  static <E extends Enum<E> & BaseEnum<T>, T extends Serializable> E getEnumConstants(
      T value, Class<E> clazz) {
    final E[] clazzEnumConstants = clazz.getEnumConstants();
    final Optional<E> first =
        Arrays.stream(clazzEnumConstants)
            .filter(e -> Objects.equals(e.getValue(), value))
            .findFirst();
    return first.orElse(null);
  }

  /**
   * 将枚举转换为选项列表.
   *
   * @param baseEnum 枚举类
   * @param clazz 选项类
   * @return 选项列表
   */
  static List<? extends Options> convertOptions(
      Class<? extends BaseEnum<?>> baseEnum, Class<? extends Options> clazz) {
    if (Objects.isNull(baseEnum) || !baseEnum.isEnum() || Objects.isNull(clazz)) {
      return List.of();
    }
    return Arrays.stream(baseEnum.getEnumConstants())
        .map(
            e -> {
              Options options = clazz.cast(Options.class);
              options.setValue(Objects.toString(e.getValue(), ""));
              options.setLabel(e.getDescription());
              options.setDefault(e.isDefault());
              return options;
            })
        .toList();
  }

  /**
   * 获取枚举描述.
   *
   * @return 描述信息
   */
  String getDescription();

  /**
   * 判断是否为默认值.
   *
   * @return true表示是默认值，false表示不是默认值
   */
  default boolean isDefault() {
    return false;
  }
}
