package com.codeupsoft.base.common.utils;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.TypeReference;
import jakarta.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerMapping;

/** 适配 Spring 6.x + Fastjson2 的 SPEL 解析工具 修复：NPE 防护、#body.username 为 null、大小写不敏感 */
@Slf4j
@SuppressWarnings("all")
public class InterceptorSpelKeyParser {
  // 全局 SPEL 解析器（线程安全）
  private static final ExpressionParser SPEL_PARSER = new SpelExpressionParser();
  // 参数名发现器（Spring 6.x 推荐）
  private static final DefaultParameterNameDiscoverer PARAM_DISCOVERER =
      new DefaultParameterNameDiscoverer();

  /** 核心方法：解析 SPEL 生成最终 Key */
  public static String parseKey(
      HttpServletRequest request, HandlerMethod handlerMethod, String spelKey) {
    try {
      HttpServletRequest requestToUse = request;
      String bodyStr = "";

      // 仅 JSON 请求包装 request，支持重复读取
      if (isJsonRequest(request)) {
        requestToUse = new RepeatableReadRequestWrapper(request);
        bodyStr = ((RepeatableReadRequestWrapper) requestToUse).getBodyAsString().trim();
      }

      // 构建 SPEL 上下文（注入所有请求变量）
      StandardEvaluationContext context =
          buildEvaluationContext(requestToUse, handlerMethod, bodyStr);

      return parseSpel(spelKey, context, "");
    } catch (Exception e) {
      log.error(e.getMessage(), e);
      return "";
    }
  }

  /** 构建 SPEL 执行上下文，注入所有请求相关变量 */
  private static StandardEvaluationContext buildEvaluationContext(
      HttpServletRequest request, HandlerMethod handlerMethod, String bodyStr) {
    StandardEvaluationContext context = new StandardEvaluationContext();

    // 1. 原生 HttpServletRequest
    context.setVariable("request", request);

    // 2. 路径变量（/user/{id} → #pathVars.id）
    Map<String, String> pathVars =
        (Map<String, String>) request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
    context.setVariable("pathVars", pathVars == null ? Collections.emptyMap() : pathVars);

    // 3. 请求头（#headers.Token → 获取 Token 请求头）
    Map<String, String> headers = new HashMap<>();
    Enumeration<String> headerNames = request.getHeaderNames();
    if (headerNames != null) {
      while (headerNames.hasMoreElements()) {
        String headerName = headerNames.nextElement();
        if (headerName == null || headerName.trim().isEmpty()) {
          continue;
        }
        headers.put(headerName, Optional.ofNullable(request.getHeader(headerName)).orElse(""));
      }
    }
    context.setVariable("headers", headers);

    // 4. 查询参数（#params.type → 获取 ?type=xxx）
    Map<String, String> params = new HashMap<>();
    Map<String, String[]> paramMap = request.getParameterMap();
    if (paramMap != null && !paramMap.isEmpty()) {
      for (Map.Entry<String, String[]> entry : paramMap.entrySet()) {
        String key = entry.getKey();
        String[] values = entry.getValue();
        if (key == null || key.trim().isEmpty() || values == null || values.length == 0) {
          continue;
        }
        params.put(key, values[0]);
      }
    }
    context.setVariable("params", params);

    // 5. JSON 请求体（核心修复：解析为普通 Map，支持大小写不敏感）
    Map<String, Object> bodyMap = new HashMap<>();
    if (!bodyStr.isEmpty() && bodyStr.startsWith("{") && bodyStr.endsWith("}")) {
      try {
        // Fastjson2 解析为普通 Map（SPEL 可直接通过 #body.username 读取）
        bodyMap = JSON.parseObject(bodyStr, new TypeReference<Map<String, Object>>() {});
      } catch (Exception e) {
        System.out.println("【调试】JSON 解析失败：" + e.getMessage());
      }
    }
    context.setVariable("bodyStr", bodyStr);
    context.setVariable("body", bodyMap);
    context.setVariable("bodyIgnoreCase", new CaseInsensitiveMap(bodyMap));

    // 6. 兼容方法参数名（#userId → 自动映射路径/查询/请求体参数）
    Method method = handlerMethod.getMethod();
    String[] paramNames = PARAM_DISCOVERER.getParameterNames(method);
    if (paramNames != null && paramNames.length > 0) {
      Map<String, String> pathVarsMap = (Map<String, String>) context.lookupVariable("pathVars");
      Map<String, String> queryParamsMap = (Map<String, String>) context.lookupVariable("params");

      for (String paramName : paramNames) {
        if (paramName == null || paramName.trim().isEmpty()) {
          continue;
        }
        // 优先级：路径变量 → 查询参数 → JSON 请求体
        Object paramValue = pathVarsMap.get(paramName);
        if (paramValue == null) {
          paramValue = queryParamsMap.get(paramName);
        }
        if (paramValue == null) {
          paramValue = bodyMap.get(paramName);
        }

        // 注入非空参数到 SPEL 上下文
        if (paramValue != null) {
          context.setVariable(paramName, paramValue);
        }
      }
    }

    return context;
  }

  /** 通用 SPEL 解析方法（带默认值） */
  private static String parseSpel(
      String spel, StandardEvaluationContext context, String defaultValue) {
    if (spel == null || spel.trim().isEmpty()) {
      return "";
    }
    try {
      Object value = SPEL_PARSER.parseExpression(spel).getValue(context);
      return value == null ? defaultValue : value.toString();
    } catch (Exception e) {
      return defaultValue;
    }
  }

  /** 判断是否为 JSON 请求（Content-Type 包含 application/json） */
  public static boolean isJsonRequest(HttpServletRequest request) {
    String contentType = request.getContentType();
    return contentType != null && contentType.contains("application/json");
  }

  /** 内部类：大小写不敏感的 Map（解决 UserName/username 字段名问题） */
  public static class CaseInsensitiveMap extends HashMap<String, Object> {
    private final Map<String, String> keyMap = new HashMap<>();

    public CaseInsensitiveMap(Map<String, Object> source) {
      super(source);
      // 存储小写 key → 原始 key 的映射
      for (String key : source.keySet()) {
        if (key != null) {
          keyMap.put(key.toLowerCase(), key);
        }
      }
    }

    @Override
    public Object get(Object key) {
      if (key instanceof String) {
        String lowerKey = ((String) key).toLowerCase();
        String originalKey = keyMap.get(lowerKey);
        if (originalKey != null) {
          return super.get(originalKey);
        }
      }
      return super.get(key);
    }
  }
}
