package com.codeupsoft.base.common.utils;

import java.io.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Random;

/** MD5 工具类，支持字符串、字节数组、文件的 MD5 计算，以及带盐值加密. */
@SuppressWarnings("all")
public class Md5Utils {

  // MD5 算法名称
  private static final String MD5_ALGORITHM = "MD5";

  // 十六进制字符集（小写）
  private static final char[] HEX_CHARS = "0123456789abcdef".toCharArray();

  // 私有构造方法，防止实例化
  private Md5Utils() {}

  /**
   * 计算字符串的 MD5（默认 UTF-8 编码）.
   *
   * @param str 待加密的字符串
   * @return MD5 十六进制字符串（32位小写）
   */
  public static String md5(String str) {
    if (str == null) {
      return null;
    }
    return md5(str.getBytes());
  }

  /**
   * 计算字符串的 MD5（指定字符集）.
   *
   * @param str 待加密的字符串
   * @param charset 字符集（如 "UTF-8"、"GBK"）
   * @return MD5 十六进制字符串（32位小写）
   * @throws UnsupportedEncodingException 字符集不支持
   */
  public static String md5(String str, String charset) throws UnsupportedEncodingException {
    if (str == null || charset == null) {
      return null;
    }
    return md5(str.getBytes(charset));
  }

  /**
   * 计算字节数组的 MD5.
   *
   * @param bytes 待加密的字节数组
   * @return MD5 十六进制字符串（32位小写）
   */
  public static String md5(byte[] bytes) {
    if (bytes == null || bytes.length == 0) {
      return null;
    }

    try {
      // 获取 MD5 消息摘要实例
      MessageDigest md = MessageDigest.getInstance(MD5_ALGORITHM);
      // 更新摘要（传入字节数组）
      md.update(bytes);
      // 计算最终摘要值（16字节数组）
      byte[] digest = md.digest();
      // 转换为十六进制字符串
      return bytesToHex(digest);
    } catch (NoSuchAlgorithmException e) {
      // MD5 是标准算法，理论上不会抛出此异常
      throw new RuntimeException("MD5 algorithm not found", e);
    }
  }

  /**
   * 计算文件的 MD5（大文件友好，分块读取）.
   *
   * @param file 待计算的文件
   * @return MD5 十六进制字符串（32位小写）
   * @throws IOException 文件读取异常
   */
  public static String md5(File file) throws IOException {
    if (file == null || !file.exists() || !file.canRead()) {
      return null;
    }

    try (InputStream in = new FileInputStream(file)) {
      MessageDigest md = MessageDigest.getInstance(MD5_ALGORITHM);
      byte[] buffer = new byte[8192]; // 8KB 缓冲区（可调整）
      int len;
      // 分块读取文件并更新摘要
      while ((len = in.read(buffer)) != -1) {
        md.update(buffer, 0, len);
      }
      return bytesToHex(md.digest());
    } catch (NoSuchAlgorithmException e) {
      throw new RuntimeException("MD5 algorithm not found", e);
    }
  }

  /**
   * 带盐值的 MD5 加密（盐值随机生成）.
   *
   * @param str 待加密的字符串
   * @return 数组：[0] 盐值（十六进制），[1] 加密后的 MD5
   */
  public static String[] md5WithSalt(String str) {
    if (str == null) {
      return null;
    }
    // 生成 8 字节随机盐值（可调整长度）
    byte[] salt = generateSalt(8);
    // 盐值转换为十六进制（方便存储）
    String saltHex = bytesToHex(salt);
    // 字符串 + 盐值 拼接后加密
    String md5Hex = md5(str + saltHex);
    return new String[] {saltHex, md5Hex};
  }

  /**
   * 带盐值的 MD5 加密（自定义盐值）.
   *
   * @param str 待加密的字符串
   * @param salt 盐值（字节数组）
   * @return 加密后的 MD5 十六进制字符串
   */
  public static String md5WithSalt(String str, byte[] salt) {
    if (str == null || salt == null) {
      return null;
    }
    // 字符串字节数组 + 盐值字节数组 拼接
    byte[] strBytes = str.getBytes();
    byte[] combined = new byte[strBytes.length + salt.length];
    System.arraycopy(strBytes, 0, combined, 0, strBytes.length);
    System.arraycopy(salt, 0, combined, strBytes.length, salt.length);
    return md5(combined);
  }

  /**
   * 验证带盐值的 MD5（用于密码校验等场景）.
   *
   * @param originalStr 原始字符串（如用户输入的密码）
   * @param saltHex 存储的盐值（十六进制）
   * @param md5Hex 存储的加密后 MD5
   * @return 验证通过返回 true，否则 false
   */
  public static boolean verifyWithSalt(String originalStr, String saltHex, String md5Hex) {
    if (originalStr == null || saltHex == null || md5Hex == null) {
      return false;
    }
    // 原始字符串 + 盐值 加密后与存储的 MD5 对比
    String calculatedMd5 = md5(originalStr + saltHex);
    return md5Hex.equals(calculatedMd5);
  }

  /**
   * 生成随机盐值.
   *
   * @param length 盐值长度（字节数，建议 8-16 字节）
   * @return 随机盐值字节数组
   */
  private static byte[] generateSalt(int length) {
    if (length <= 0) {
      throw new IllegalArgumentException("Salt length must be positive");
    }
    byte[] salt = new byte[length];
    new Random().nextBytes(salt);
    return salt;
  }

  /**
   * 字节数组转换为十六进制字符串（小写）.
   *
   * @param bytes 字节数组
   * @return 十六进制字符串
   */
  private static String bytesToHex(byte[] bytes) {
    if (bytes == null || bytes.length == 0) {
      return null;
    }
    char[] chars = new char[bytes.length * 2];
    for (int i = 0; i < bytes.length; i++) {
      int b = bytes[i] & 0xFF; // 转换为无符号字节
      chars[i * 2] = HEX_CHARS[b >>> 4]; // 高4位
      chars[i * 2 + 1] = HEX_CHARS[b & 0x0F]; // 低4位
    }
    return new String(chars);
  }
}
