package com.codeupsoft.base.common.utils;

import jakarta.servlet.ReadListener;
import jakarta.servlet.ServletInputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

/** 包装HttpServletRequest，支持重复读取请求体（适配拦截器解析JSON请求体） */
public class RepeatableReadRequestWrapper extends HttpServletRequestWrapper {
  private final byte[] body;

  public RepeatableReadRequestWrapper(HttpServletRequest request) throws IOException {
    super(request);
    // 读取原始请求体到字节数组
    body = readBytes(request.getInputStream());
  }

  // 读取输入流为字节数组
  private byte[] readBytes(InputStream inputStream) throws IOException {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    byte[] buffer = new byte[1024];
    int len;
    while ((len = inputStream.read(buffer)) != -1) {
      outputStream.write(buffer, 0, len);
    }
    return outputStream.toByteArray();
  }

  @Override
  public ServletInputStream getInputStream() throws IOException {
    // 返回包装后的输入流（可重复读取）
    ByteArrayInputStream inputStream = new ByteArrayInputStream(body);
    return new ServletInputStream() {
      @Override
      public boolean isFinished() {
        return inputStream.available() == 0;
      }

      @Override
      public boolean isReady() {
        return true;
      }

      @Override
      public void setReadListener(ReadListener listener) {}

      @Override
      public int read() throws IOException {
        return inputStream.read();
      }
    };
  }

  // 获取请求体字符串
  public String getBodyAsString() {
    return new String(body);
  }
}
