package com.codeupsoft.base.common.utils;

import cn.hutool.core.convert.Convert;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import lombok.experimental.UtilityClass;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * 请求工具类，提供获取当前请求上下文相关信息的便捷方法.
 *
 * @author ruoyi
 */
@UtilityClass
public class RequestUtils {

  /** 获取String参数. */
  public static String getParameter(String name) {
    return getRequest().getParameter(name);
  }

  /** 获取String参数. */
  public static String getParameter(String name, String defaultValue) {
    return Convert.toStr(getRequest().getParameter(name), defaultValue);
  }

  /** 获取Integer参数. */
  public static Integer getParameterToInt(String name) {
    return Convert.toInt(getRequest().getParameter(name));
  }

  /** 获取Integer参数. */
  public static Integer getParameterToInt(String name, Integer defaultValue) {
    return Convert.toInt(getRequest().getParameter(name), defaultValue);
  }

  /** 获取Boolean参数. */
  public static Boolean getParameterToBool(String name) {
    return Convert.toBool(getRequest().getParameter(name));
  }

  /** 获取Boolean参数. */
  public static Boolean getParameterToBool(String name, Boolean defaultValue) {
    return Convert.toBool(getRequest().getParameter(name), defaultValue);
  }

  /**
   * 获得所有请求参数.
   *
   * @param request 请求对象{@link ServletRequest}
   * @return Map
   */
  public static Map<String, String[]> getParams(ServletRequest request) {
    final Map<String, String[]> map = request.getParameterMap();
    return Collections.unmodifiableMap(map);
  }

  /**
   * 获得所有请求参数.
   *
   * @param request 请求对象{@link ServletRequest}
   * @return Map
   */
  public static Map<String, String> getParamMap(ServletRequest request) {
    Map<String, String> params = new HashMap<>();
    for (Map.Entry<String, String[]> entry : getParams(request).entrySet()) {
      params.put(entry.getKey(), StringUtils.join(entry.getValue(), ","));
    }
    return params;
  }

  /** 获取request. */
  public static HttpServletRequest getRequest() {
    return getRequestAttributes().getRequest();
  }

  /** 获取response. */
  public static HttpServletResponse getResponse() {
    return getRequestAttributes().getResponse();
  }

  /** 获取session. */
  public static HttpSession getSession() {
    return getRequest().getSession();
  }

  /** 获取ServletRequestAttributes. */
  public static ServletRequestAttributes getRequestAttributes() {
    RequestAttributes attributes = RequestContextHolder.getRequestAttributes();
    return (ServletRequestAttributes) attributes;
  }

  /**
   * 内容编码.
   *
   * @param str 内容
   * @return 编码后的内容
   */
  public static String urlEncode(String str) {
    return URLEncoder.encode(str, StandardCharsets.UTF_8);
  }

  /**
   * 内容解码.
   *
   * @param str 内容
   * @return 解码后的内容
   */
  public static String urlDecode(String str) {
    return URLDecoder.decode(str, StandardCharsets.UTF_8);
  }

  /**
   * 获取请求头.
   *
   * @param request 请求对象
   * @return 请求头
   */
  public static Map<String, String> getHeaders(HttpServletRequest request) {
    Map<String, String> map = new HashMap<>();
    final Enumeration<String> names = request.getHeaderNames();
    while (names.hasMoreElements()) {
      final String name = names.nextElement();
      map.put(name, request.getHeader(name));
    }
    return map;
  }
}
