package com.codeupsoft.interceptor.xss.autoconfigure;

import com.codeupsoft.interceptor.xss.core.XssInterceptor;
import com.codeupsoft.interceptor.xss.handler.XssClearHandler;
import com.codeupsoft.interceptor.xss.properties.XssProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

/**
 * XSS防护自动配置类.
 *
 * <p>该配置类负责注册XSS防护拦截器，当满足条件时自动生效.
 *
 * @author codeupsoft
 */
@Slf4j
@AutoConfiguration
@EnableConfigurationProperties(XssProperties.class)
@RequiredArgsConstructor
@ConditionalOnProperty(
    prefix = "codeupsoft.interceptor.xss",
    value = "auto-configure",
    havingValue = "true",
    matchIfMissing = true)
public class XssAutoConfiguration implements WebMvcConfigurer {

  private final XssProperties xssProperties;
  private final ApplicationContext applicationContext;

  @Override
  public void addInterceptors(InterceptorRegistry registry) {
    XssClearHandler xssClearHandler = new XssClearHandler();
    try {
      xssClearHandler = applicationContext.getBean(XssClearHandler.class);
    } catch (Exception e) {
      log.info("XssClearHandler not found, using default");
    }
    final XssInterceptor interceptor = new XssInterceptor(xssProperties, xssClearHandler);
    log.info("XSS protection interceptor added");
    registry
        .addInterceptor(interceptor)
        .addPathPatterns("/**")
        .excludePathPatterns(xssProperties.getIgnoreUrls());
  }
}
