package com.codeupsoft.interceptor.xss.core;

import com.codeupsoft.interceptor.xss.annotation.XssIgnore;
import com.codeupsoft.interceptor.xss.handler.XssClearHandler;
import com.codeupsoft.interceptor.xss.properties.XssProperties;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.util.List;
import lombok.RequiredArgsConstructor;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

/**
 * XSS防护拦截器.
 *
 * <p>用于拦截HTTP请求，对请求参数进行XSS过滤.
 *
 * @author codeupsoft
 */
@RequiredArgsConstructor
public class XssInterceptor implements HandlerInterceptor {

  private final XssProperties xssProperties;
  private final XssClearHandler xssClearHandler;

  @Override
  public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
      throws Exception {
    // 如果不是方法处理器，直接放行
    if (!(handler instanceof HandlerMethod handlerMethod)) {
      return true;
    }

    // 检查是否有XssIgnore注解，如果有则忽略XSS检查
    if (handlerMethod.hasMethodAnnotation(XssIgnore.class)) {
      return true;
    }

    // 检查是否启用XSS防护
    if (!xssProperties.isEnabled()) {
      return true;
    }

    // 检查请求方法是否在忽略列表中
    String method = request.getMethod();
    List<String> ignoreMethods = xssProperties.getIgnoreMethods();
    if (ignoreMethods != null && ignoreMethods.contains(method)) {
      return true;
    }

    RequestContextHolder.setRequestAttributes(
        new ServletRequestAttributes(new XssHttpServletRequestWrapper(request, xssClearHandler)));

    return true;
  }
}
