package com.codeupsoft.interceptor.xss.handler;

import org.springframework.web.util.HtmlUtils;

/**
 * XSS清理处理器.
 *
 * <p>负责对字符串进行XSS清理处理.
 *
 * @author codeupsoft
 */
@SuppressWarnings("unused")
public class XssClearHandler {

  /**
   * 清理XSS攻击代码.
   *
   * <p>对输入的字符串进行HTML转义，以防止XSS攻击.
   *
   * @param value 需要清理的原始字符串
   * @return 清理后的安全字符串
   */
  public String clearXss(String value) {
    return reply(HtmlUtils.htmlEscape(value));
  }

  /**
   * 回复特殊字符.
   *
   * <p>将HTML转义后的字符串中的&amp;符号还原为&amp;，以便于JSON等格式的数据处理.
   *
   * @param value 经过HTML转义的字符串
   * @return 还原后的字符串
   */
  public String reply(String value) {
    // json还原
    value = value.replace("&amp;", "&");
    value = value.replace("&quot;", "\"");

    return value;
  }

  /**
   * 恢复数据.
   *
   * <p>对经过HTML转义的字符串进行反向操作，恢复成原始字符串.
   *
   * @param value 经过HTML转义的字符串
   * @return 原始字符串
   */
  public String restoreData(String value) {
    return HtmlUtils.htmlUnescape(value);
  }
}
