package com.ctaiot;

import com.ctaiot.bean.*;
import com.ctaiot.connection.HttpProxy;
import com.ctaiot.print.PrintClient;
import com.ctaiot.resp.APIConnectionException;
import com.ctaiot.resp.APIRequestException;
import com.ctaiot.utils.BeanUtil;
import com.ctaiot.utils.EncryptionMD5;

import java.beans.IntrospectionException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class CloudPrintClient {

    private PrintClient printClient;

    private EncryptionMD5 encryptionMD5;

    private CloudPrintClient() {
    }

    public CloudPrintClient(String appKey, String appSecret) {
        printClient = new PrintClient(appKey, appSecret);
        encryptionMD5 = new EncryptionMD5();
    }

    public CloudPrintClient(String appKey, String appSecret, CloudPrintConfig config) {
        printClient = new PrintClient(appKey, appSecret, config);
        encryptionMD5 = new EncryptionMD5();
    }

    public CloudPrintClient(String appKey, String appSecret, HttpProxy proxy, CloudPrintConfig config) {
        printClient = new PrintClient(appKey, appSecret, proxy, config);
        encryptionMD5 = new EncryptionMD5();
    }

    /**
     * create sdk token
     * <p>
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     * @Param userId 发起方系统用户标志 只支持英文大小写，数字，长度不超过25位
     */
    public PrintResult getPrintToken(String userId) throws APIConnectionException, APIRequestException {
        return printClient.getSdkToken(userId);
    }

    /**
     * get brand list
     * <p>
     * brand
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public List<BrandResult> getPrintBrand(Brand brand, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
//        if (Objects.isNull(brand.getEquip())) {
//            throw new IllegalArgumentException("equip should not be null");
//        }
        brand.setEquip(null);
        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(brand), brand.getEquip());

        return printClient.getBrandList(payLoad).fromDataList(BrandResult.class);
    }

    /**
     * create brand
     * <p>
     * payLoad
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult savePrintBrand(BrandPayLoad payLoad, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
        PrintPayLoad printPayLoad = this.setPayLoad(token, BeanUtil.beanToMap(payLoad), payLoad.getEquip());

        return printClient.saveBrand(printPayLoad);
    }

    /**
     * get folder label list
     * <p>
     * folderLabel
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public FolderPlugCloudResult getFolderLabelList(FolderLabel folderLabel, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(folderLabel), folderLabel.getEquip());

        return printClient.getFolderLabelList(payLoad).fromData(FolderPlugCloudResult.class);
    }

    /**
     * get clientId
     * <p>
     * equip
     * token
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult getPrintClientId(Integer equip, String token) throws APIConnectionException, APIRequestException {
        Map<String, Object> params = new HashMap<>();
        params.put("equip", equip);
        PrintPayLoad payLoad = this.setPayLoad(token, params, equip);

        return printClient.getClientIdInfo(payLoad);
    }

    /**
     * send api print
     * <p>
     * apiPrintPayLoad
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult sendApiPrint(ApiPrintPayLoad apiPrintPayLoad, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
        if (apiPrintPayLoad.getClientId() == null || apiPrintPayLoad.getClientId().isEmpty()) {
            throw new IllegalArgumentException("clientId should not be null");
        }

        Map<String, Object> params = BeanUtil.beanToMap(apiPrintPayLoad, false, true);
        params.put("cloud_doc_md5", encryptionMD5.getMD5ofStr(apiPrintPayLoad.getCloudDoc()));

        PrintPayLoad payLoad = this.setPayLoad(token, params, apiPrintPayLoad.getEquip());

        return printClient.sendApiPrint(payLoad);
    }


    /**
     * send pdf print
     * <p>
     * pdfPrintPayLoad
     *
     * @throws APIConnectionException
     * @throws
     */
    public PrintResult sendPdfPrint(PdfPrintPayLoad pdfPrintPayLoad, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
        if (pdfPrintPayLoad.getClientId() == null || pdfPrintPayLoad.getClientId().isEmpty()) {
            throw new IllegalArgumentException("clientId should not be null");
        }

        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(pdfPrintPayLoad, false, true), pdfPrintPayLoad.getEquip());

        return printClient.sendPdfPrint(payLoad);
    }

    /**
     * send pic print
     * <p>
     * picPrintPayLoad
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult sendPicPrint(PicPrintPayLoad picPrintPayLoad, String token) throws APIConnectionException, APIRequestException, IntrospectionException, InvocationTargetException, IllegalAccessException {
        if (picPrintPayLoad.getClientId() == null || picPrintPayLoad.getClientId().isEmpty()) {
            throw new IllegalArgumentException("clientId should not be null");
        }

        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(picPrintPayLoad, false, true), picPrintPayLoad.getEquip());
        return printClient.sendPicPrint(payLoad);
    }

    /**
     * get print url drives
     * <p>
     * token
     *
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult getPrintDownUrl(String token) throws APIConnectionException, APIRequestException {
        PrintPayLoad payLoad = this.setPayLoad(token, null, null);
        return printClient.printUrl(payLoad);
    }

    /**
     * get print task list
     * taskPayLoad
     * token
     *
     * @throws IntrospectionException
     * @throws InvocationTargetException
     * @throws IllegalAccessException
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public TaskPageResult getPrintTask(TaskPayLoad taskPayLoad, String token) throws IntrospectionException, InvocationTargetException, IllegalAccessException, APIConnectionException, APIRequestException {
        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(taskPayLoad, false, true), taskPayLoad.getEquip());

        return printClient.listPrintTask(payLoad).fromData(TaskPageResult.class);
    }

    /**
     * operate print task
     * taskPayLoad
     * token
     *
     * @throws IntrospectionException
     * @throws InvocationTargetException
     * @throws IllegalAccessException
     * @throws APIConnectionException
     * @throws APIRequestException
     */
    public PrintResult operatePrintTask(OperateTaskPayLoad taskPayLoad, String token) throws IntrospectionException, InvocationTargetException, IllegalAccessException, APIConnectionException, APIRequestException {
        PrintPayLoad payLoad = this.setPayLoad(token, BeanUtil.beanToMap(taskPayLoad, false, false), null);

        return printClient.operatePrintTask(payLoad);
    }

    /**
     * request params set
     * <p>
     * token
     * params
     * equip
     */
    private PrintPayLoad setPayLoad(String token, Map<String, Object> params, Integer equip) {
        PrintPayLoad payLoad = new PrintPayLoad();

        if (params != null && !params.isEmpty()) {
            payLoad.setParams(params);
        }

        Map<String, Object> map = new HashMap<>();
        map.put("equip", equip);
        map.put("accessToken", token);

        payLoad.setHeaders(map);

        return payLoad;
    }
}